//
#include <Shaders/VisionCommon.inc>
#include <Shaders/TextureSamplingHelpers.inc>
#include <Shaders/DeferredShadingHelpers.inc>

#define EPSILON 0.00001f
#define PI 3.1415926535897932384626433832795f

struct PS_IN
{
  float4   ProjPos    : SV_Position;
  
  #if defined(_VISION_PS3) || defined(_VISION_PSP2)
    float2 ScreenPos : WPOS;
  #elif defined(_VISION_DX11)
    float4 ScreenPos : VPOS_MANUAL;
  #elif defined(_VISION_XENON)
    float4 ScreenPos : TEXCOORD0;
  #else
    float2 ScreenPos : VPOS;
  #endif
  
  float3 UV0 : TEXCOORD1;
  

  float3 ObjPos : TEXCOORD2;
};

#ifdef _VISION_DX11
cbuffer g_GlobalConstantBufferUser : register (b2)
{
  float3 SunDirection    : packoffset(c5);
  
  float3 UpperLeftCorner : packoffset(c17);
  float3 RightDir        : packoffset(c18);
  float3 DownDir         : packoffset(c19);  
  float4 InvTexSizeMask  : packoffset(c20); // xy = inv. screen size, zw = inv. scatter texture size
  
  // Array with pairs of 2 registers for every layer:
  //  reg[2n+0] : x:alpha, y:sphere curvedness, z:sphere height offset
  //  reg[2n+1] : xy: texture tiling, zw: texture scroll ofs
  float4 SkyTransform[8] : packoffset(c21);  
}

#elif defined(_VISION_PS3) || defined(_VISION_PSP2) || defined(_VISION_GLES2) || defined(_VISION_WIIU)

  float3 SunDirection          : register(c37);

  float3 UpperLeftCorner       : register(c50);
  float3 RightDir              : register(c51);
  float3 DownDir               : register(c52);
  float4 InvTexSizeMask        : register(c53);
  
  float4   SkyTransform[8] : register(c54);
#else

  float3 SunDirection;
  
  float3 UpperLeftCorner;
  float3 RightDir;
  float3 DownDir;
  float4 InvTexSizeMask;
  float4 SkyTransform[8] : register(c40);
#endif


#ifdef _VISION_DX11
  Texture2D <float4> TransmittanceTexture : register(t0);
  sampler TransmittanceSampler            : register(s0);
  
  Texture2D <float4> CloudTransmittanceTexture : register(t1);
  sampler CloudTransmittanceSampler            : register(s1);
#else
  sampler2D TransmittanceTexture          : register(s0);
  sampler2D CloudTransmittanceTexture     : register(s1);
#endif

// Cloud samplers
#ifdef _VISION_DX11

  Texture2D <float4>    CloudTexture0         : register(t3);
  sampler               CloudTextureSampler0  : register(s3);

  Texture2D <float4>    CloudTexture1         : register(t4);
  sampler               CloudTextureSampler1  : register(s4);

  Texture2D <float4>    CloudTexture2         : register(t5);
  sampler               CloudTextureSampler2  : register(s5);

  Texture2D <float4>    CloudTexture3         : register(t6);
  sampler               CloudTextureSampler3  : register(s6);

#else

  sampler2D             CloudTexture0  : register(s3);
  sampler2D             CloudTexture1  : register(s4);
  sampler2D             CloudTexture2  : register(s5);
  sampler2D             CloudTexture3  : register(s6);

#endif


struct SkyCoords
{
  float2 sph;
  float3 dir;
  float2 uv;
};

float2 ComputeSphericalCoords(PS_IN In, float4 transform, float2 params)
{
  float3 pos = In.ObjPos;
  pos.z *= params.x; // curvedness
  pos = normalize(pos);
  pos.z += params.y; // height ofs
  pos = normalize(pos);
  return pos.yx*transform.xy+transform.zw; // scale and scroll (swap xy for backwards compatibility)
}

float4 SampleClouds0(SkyCoords coords, PS_IN In)
{
$ifdef LAYER0_CUBE
  return vTex2D(CloudTexture0, CloudTextureSampler0, coords.uv);
$elif LAYER0_SPHERE
  return vTex2D(CloudTexture0, CloudTextureSampler0, ComputeSphericalCoords(In, SkyTransform[1], SkyTransform[0].yz));
$elif LAYER0_CYLINDER
  return vTex2D(CloudTexture0, CloudTextureSampler0, float2(coords.sph.x, saturate(0.5f-coords.dir.z+SkyTransform[0].z)));
$endif
}

float4 SampleClouds1(SkyCoords coords, PS_IN In)
{
$ifdef LAYER1_CUBE
  return vTex2D(CloudTexture1, CloudTextureSampler1, coords.uv);
$elif LAYER1_SPHERE
  return vTex2D(CloudTexture1, CloudTextureSampler1, ComputeSphericalCoords(In, SkyTransform[3], SkyTransform[2].yz));
$elif LAYER1_CYLINDER
  return vTex2D(CloudTexture1, CloudTextureSampler1, float2(coords.sph.x, saturate(0.5f-coords.dir.z+SkyTransform[2].z)));
$endif
}

float4 SampleClouds2(SkyCoords coords, PS_IN In)
{
$ifdef LAYER2_CUBE
  return vTex2D(CloudTexture2, CloudTextureSampler2, coords.uv);
$elif LAYER2_SPHERE
  return vTex2D(CloudTexture2, CloudTextureSampler2, ComputeSphericalCoords(In, SkyTransform[5], SkyTransform[4].yz));
$elif LAYER2_CYLINDER
  return vTex2D(CloudTexture2, CloudTextureSampler2, float2(coords.sph.x, saturate(0.5f-coords.dir.z+SkyTransform[4].z)));
$endif
}

float4 SampleClouds3(SkyCoords coords, PS_IN In)
{
$ifdef LAYER3_CUBE
  return vTex2D(CloudTexture3, CloudTextureSampler3, coords.uv);
$elif LAYER3_SPHERE
  return vTex2D(CloudTexture3, CloudTextureSampler3, ComputeSphericalCoords(In, SkyTransform[7], SkyTransform[6].yz));
$elif LAYER3_CYLINDER
  return vTex2D(CloudTexture3, CloudTextureSampler3, float2(coords.sph.x, saturate(0.5f-coords.dir.z+SkyTransform[6].z)));
$endif
}



float RayleighPhase(float cosTheta)
{
  return 0.75f * (1 + cosTheta * cosTheta);
}

float MiePhase(float cosTheta, float g)
{
  return (3 * (1 - g*g)) / (2 * (2 + g*g)) * (1 + cosTheta * cosTheta) / pow((1 + g*g - 2 * g * cosTheta), 1.5f);
}

SkyCoords GetSkyCoords(float3 dir, float2 uv)
{
  SkyCoords q;
  q.dir = dir;
  
  float phi = atan2(dir.y, dir.x);
  float theta = acos(min(0.9999f,dir.z));
    
  q.sph = float2(phi / (2 * PI) + 0.5f, theta / PI);
  q.uv = uv;
  return q;
}

// Bilinear floating point texture sampling not supported on Xbox360 and PS3
#if defined(_VISION_XENON) || defined(_VISION_PS3)
  #define SampleFloatingPointTextureBilinear(tex, sampler, texcoords, invTexSize) \
    SampleBilinear(tex, sampler, texcoords / invTexSize, invTexSize)
#else
  #define SampleFloatingPointTextureBilinear(tex, sampler, texcoords, invTexSize) \
    vTex2D(tex, sampler, texcoords);
#endif

float4 SampleSky(float3 dir, PS_IN In)
{
  SkyCoords skyCoords = GetSkyCoords(dir, In.UV0);
  
  float4 s1 = SampleFloatingPointTextureBilinear(TransmittanceTexture, TransmittanceSampler, skyCoords.sph, InvTexSizeMask.zw);
  
$if LAYER_COUNT == 0
  return s1;
$else

  float4 s2 = SampleFloatingPointTextureBilinear(CloudTransmittanceTexture, CloudTransmittanceSampler, skyCoords.sph, InvTexSizeMask.zw);
  
  float4 clouds;
  float t;
  
  float sum = SkyTransform[0].x;
  clouds = SampleClouds0(skyCoords, In) * SkyTransform[0].x;

$if LAYER_COUNT > 1
  clouds += SampleClouds1(skyCoords, In) * SkyTransform[2].x;
  sum += SkyTransform[2].x;
$endif

$if LAYER_COUNT > 2
  clouds += SampleClouds2(skyCoords, In) * SkyTransform[4].x;
  sum += SkyTransform[4].x;
$endif

$if LAYER_COUNT > 3
  clouds += SampleClouds3(skyCoords, In) * SkyTransform[6].x;
  sum += SkyTransform[6].x;
$endif

  if (sum > 0)
    clouds /= sum;

  t = 1-exp(-clouds.w);

  float4 scattering = (s2-s1)*t + s1;
  return 0.5f*(scattering + dot(clouds.xyz, float3(1,1,1)/3));
  
$endif
}

float2 GetScreenPos(PS_IN In)
{
  float2 screenPos;
  
  #if defined(_VISION_DX11)
    screenPos = (In.ScreenPos.xy / In.ScreenPos.w) * 0.5f + 0.5f;
  #elif defined(_VISION_XENON)
    // put InvTexSizeMask.xy * 0.5f (half pixel offset) inside the braces
    screenPos = (In.ScreenPos.xy / In.ScreenPos.w + InvTexSizeMask.xy) * 0.5f + 0.5f;
  #elif defined(_VISION_PS3)
    screenPos = In.ScreenPos.xy * InvTexSizeMask.xy;
    screenPos.y = 1.0f - screenPos.y;
  #else
    screenPos = (In.ScreenPos.xy + 0.5f) * InvTexSizeMask.xy;
  #endif
 
  return screenPos;
}

float4 ps_main( PS_IN In ) : SV_Target
{
  float2 screenPos = GetScreenPos(In);

  // Compute view direction in world space coordinates
  float3 ViewDirection = normalize(UpperLeftCorner + RightDir * screenPos.x + DownDir * screenPos.y);
 
  // Compute the sky color
  float4 SkyColor = SampleSky(ViewDirection, In);
      
  return float4(SkyColor.xyz, 1);
}
