#include <Shaders/VisionCommon.inc>
#include <Shaders/ShaderHelpers.inc>
#include "DeferredShadingHelpers.inc"

$ifdef DYNLIGHT_SPOT_PROJECTOR
  #define DYNLIGHT_USE_REFPLANES
$endif

$ifdef DYNLIGHT_OMNI_PROJECTOR
  #define DYNLIGHT_USE_REFPLANES
$endif

$if MSAA_SAMPLES > 1
  #define InputBuffer Texture2DMS<float4, $MSAA_SAMPLES>
  #define SampleBuffer(_tex_, _sampler_, _uv_, _slice_) _tex_.Load(_uv_, _slice_)
$else
  #define InputBuffer Texture2D<float4>
  #define SampleBuffer(_tex_, _sampler_, _uv_, _slice_) vTex2D(_tex_, _sampler_, _uv_)
$endif

$if (MSAA_SAMPLES > 1) && !defined (MSAA_SINGLESAMPLE)
  #define SAMPLE_COUNT $MSAA_SAMPLES
  #define LIGHT_CLIP(x)
$else
  #define SAMPLE_COUNT 1
  #define LIGHT_CLIP(x) clip(x)
$endif

#ifdef _VISION_DX11
  InputBuffer        DiffuseBuffer        : register(t0);
  sampler            DiffuseBufferSampler : register(s0);
  InputBuffer        NormalBuffer        : register(t1);
  sampler            NormalBufferSampler : register(s1);
  InputBuffer        DepthSpecBuffer        : register(t2);
  sampler            DepthSpecBufferSampler : register(s2);
  Texture2D <float4> AttenTex        : register(t3);
  sampler            AttenTexSampler : register(s3);
  $ifdef DYNLIGHT_OMNI_PROJECTOR
    TextureCube <float4> ProjTex        : register(t4);
    sampler              ProjTexSampler : register(s4);
  $endif
  $ifdef DYNLIGHT_SPOT_PROJECTOR
    Texture2D <float4> ProjTex        : register(t4);
    sampler            ProjTexSampler : register(s4);
  $endif
  
  $ifdef DYNLIGHT_DEFERRED_SHADOWING
    InputBuffer ShadowTex        : register(t5);
    sampler     ShadowTexSampler : register(s5);
  $endif

  cbuffer g_GlobalConstantBufferUser : register (b2)
  {
    float3 LightPos : packoffset(c0);
    float4 LightColor:packoffset(c1);
    float4 LightRadius:packoffset(c2);
    float3 LightDir : packoffset(c3);
    float4 LightAngle : packoffset(c4);

    #ifdef DYNLIGHT_USE_REFPLANES
      float4 RefPlaneX : packoffset(c5);
      float4 RefPlaneY : packoffset(c6);
      float4 RefPlaneW : packoffset(c7);
    #endif
  
		float3 UpperLeftCorner : packoffset(c17);
		float3 RightDir : packoffset(c18);
		float3 DownDir : packoffset(c19);
		float2 InvTexSizeMask : packoffset(c20);
  }

#else
  sampler2D          DiffuseBuffer : register(s0);
  sampler2D          NormalBuffer : register(s1);
  sampler2D          DepthSpecBuffer : register(s2);
  sampler2D          AttenTex : register(s3);      //Attenuation curve
  
  $ifdef DYNLIGHT_OMNI_PROJECTOR
    samplerCUBE        ProjTex : register(s4);
  $endif
  
  $ifdef DYNLIGHT_SPOT_PROJECTOR
    sampler2D          ProjTex : register(s4);
  $endif
  
  $ifdef DYNLIGHT_DEFERRED_SHADOWING
    sampler2D          ShadowTex : register(s5);
  $endif

  float3 LightPos : register(c32);
  float4 LightColor:register(c33);
  float4 LightRadius:register(c34);
  float3 LightDir : register(c35);
  float4 LightAngle : register(c36);

  #ifdef DYNLIGHT_USE_REFPLANES
    float4 RefPlaneX : register(c37);
    float4 RefPlaneY : register(c38);
    float4 RefPlaneW : register(c39);
  #endif

	float3 UpperLeftCorner : register(c50);
	float3 RightDir : register(c51);
	float3 DownDir : register(c52);
	float2 InvTexSizeMask : register(c53);

#endif

struct PS_IN                                 
{
  float4 ProjPos  : SV_Position;
  float3 EyePos : TEXCOORD2;

	#ifdef _VISION_PS3
		float2 UV0 : WPOS;
	#elif defined(_VISION_DX11)
		float4 UV0 : VPOS_MANUAL;//SV_Position;
	#else
		float2 UV0 : VPOS;
	#endif

  #if (SAMPLE_COUNT > 1)
    uint uiSampleIndex : SV_SampleIndex;
  #endif
};

float4 ComputeLighting(PS_IN In, PixelDataPacked packedData, int iSample)
{
  PixelDataUnpacked unpackedPixelData;
  unpackedPixelData = unpackPixelValues(packedData);
  
  $if !defined(DYNLIGHT_DIRECTIONAL) || defined(VLIGHT_SPECULAR)
		float3 ViewDirection = UpperLeftCorner + RightDir * In.UV0.x + DownDir * In.UV0.y;
		float3 vPos = positionForDepth(unpackedPixelData.fDepth, In.EyePos, ViewDirection);
  $endif

  $ifndef DYNLIGHT_DIRECTIONAL
    float3 vLightVec = LightPos - vPos;
    float fLightLookupU = length(vLightVec) * LightRadius.y; 
    float LightInt = vTex2D(AttenTex, AttenTexSampler, float2(fLightLookupU, 0.0f)).x;
    $if defined(DYNLIGHT_SPOT) || defined(DYNLIGHT_SPOT_PROJECTOR)
      LightInt *= saturate((dot(LightDir, -normalize(vLightVec)) - LightAngle.x) * LightAngle.y);
    $endif
  $endif

  $ifdef DYNLIGHT_OMNI_PROJECTOR
    float3 ProjTexUV;
    ProjTexUV.x = dot(vLightVec,RefPlaneX.xyz);
    ProjTexUV.y = dot(vLightVec,RefPlaneY.xyz);
    ProjTexUV.z = dot(vLightVec,RefPlaneW.xyz);
    float4 vLightColor = LightColor*vTexCUBE(ProjTex, ProjTexSampler, ProjTexUV);
  $elif defined(DYNLIGHT_SPOT_PROJECTOR) 
    float4 ProjTexUV;
    ProjTexUV.x = dot(RefPlaneX, float4(vPos, 1.0f));
    ProjTexUV.y = dot(RefPlaneY, float4(vPos, 1.0f));
    ProjTexUV.w = dot(RefPlaneW, float4(vPos, 1.0f));
    clip(ProjTexUV.w);
    ProjTexUV.xy = (ProjTexUV.xy / ProjTexUV.w) + 0.5f;
    float4 vLightColor = LightColor * vTex2D(ProjTex, ProjTexSampler, ProjTexUV.xy); 
  $else
    float4 vLightColor = LightColor;
  $endif

  $ifdef DYNLIGHT_DIRECTIONAL
    float3 vNormLightVec = -LightDir;
    float LightInt = LightColor.w;
  $else
    float3 vNormLightVec = normalize(vLightVec);
    LightInt *= LightColor.w;
  $endif

  float fDotProd = saturate(dot(vNormLightVec, unpackedPixelData.vNormal));

  // clips samples on backfacing polygons (only in non-MSAA mode)
  LIGHT_CLIP(fDotProd-0.005f);

  float4 ResColor;
  ResColor.xyz = unpackedPixelData.vDiffColor.xyz * LightInt * vLightColor.xyz * fDotProd;
  
  $ifdef VLIGHT_SPECULAR
    float3 vHalfAngle = normalize(In.EyePos - vPos) + vNormLightVec;
    float fSpecProd = GetSpecularIlluminationPoint(vHalfAngle, normalize(unpackedPixelData.vNormal), unpackedPixelData.fSpecExp * 255.f);
    fSpecProd = fSpecProd * LightInt * fDotProd;
    ResColor.xyz += unpackedPixelData.vSpecColor.xyz * fSpecProd * vLightColor.xyz;
  $endif
  
  $ifdef DYNLIGHT_DEFERRED_SHADOWING
    $if MSAA_SAMPLES > 1
      int2 uv = int2 (In.ProjPos.xy);
      float4 fShadow = ShadowTex.Load(uv, iSample);
      ResColor.xyz *= fShadow.rgb;
    $else
      #ifdef _VISION_XENON
		// Note: shadow information is only stored in the
		//   green channel, thus we avoid to overwrite the
		//   stencil values in EDRAM on xbox360 in case of
		//   interleaved shadow map rendering
        ResColor.xyz *= vTex2D(ShadowTex, ShadowTexSampler, In.UV0.xy).g;
      #else
        ResColor.xyz *= vTex2D(ShadowTex, ShadowTexSampler, In.UV0.xy).rgb;
      #endif
    $endif
  $endif

  ResColor.a = 1.0;
   
  return ResColor;
}

float4 ps_main( PS_IN In ) : SV_Target
{
	#ifdef _VISION_DX11
		In.UV0.xy = (In.UV0.xy / In.UV0.w) * 0.5f + 0.5f;
	#elif defined(_VISION_XENON)
		// On Xbox360, we need to take the tiling into account
		In.UV0.xy = (In.UV0.xy) * InvTexSizeMask + 0.5f * InvTexSizeMask;
	#elif defined(_VISION_PS3)
		In.UV0.xy = In.UV0.xy * InvTexSizeMask;
		In.UV0.y = 1.0f - In.UV0.y;
	#elif defined(_VISION_WIIU)
	    In.UV0.xy = In.UV0.xy * InvTexSizeMask;
	#else
		In.UV0.xy = In.UV0.xy * InvTexSizeMask + 0.5f * InvTexSizeMask;
	#endif

  float4 Result = float4(0,0,0,0);  

  $if MSAA_SAMPLES > 1
    int2 uv = int2(In.ProjPos.xy);

    #if (SAMPLE_COUNT == 1)
      int iSample = 0;
    #else
      int iSample = In.uiSampleIndex;
    #endif

    PixelDataPacked packedData;
    packedData.Accumulation = float4(0.0, 0.0, 0.0, 0.0);
    packedData.Diffuse = DiffuseBuffer.Load(uv, iSample);
    packedData.Normal = NormalBuffer.Load(uv, iSample);
    packedData.DepthSpec = DepthSpecBuffer.Load(uv, iSample);

  $else
    int iSample = 0;

    PixelDataPacked packedData;
    packedData.Accumulation = float4(0.0f, 0.0f, 0.0f, 0.0f);
    packedData.Diffuse = SampleBuffer(DiffuseBuffer, DiffuseBufferSampler, In.UV0.xy, 0);
    packedData.Normal = SampleBuffer(NormalBuffer, NormalBufferSampler, In.UV0.xy, 0);
    packedData.DepthSpec = SampleBuffer(DepthSpecBuffer, DepthSpecBufferSampler, In.UV0.xy, 0);
  $endif

  Result = ComputeLighting(In, packedData, iSample);

  Result.a = 1.0;

  return Result;
}


