#include <Shaders/VisionCommon.inc>
#include <Shaders/ShaderHelpers.inc>
#include "DeferredShadingHelpers.inc"

#ifdef _VISION_DX11
  Texture2D <float4> DepthSpecBuffer        : register(t0);
  sampler            DepthSpecBufferSampler : register(s0);
  Texture2D <float4> NormalBuffer           : register(t1);
  sampler            NormalBufferSampler    : register(s1);
  Texture2D <float4> NoiseNormalMap         : register(t2);
  sampler            NoiseNormalMapSampler  : register(s2);
  
  cbuffer g_GlobalConstantBufferObject : register (b1) {
		float4x4 mModelViewMatrix : packoffset(c0);
  }
  
  cbuffer g_GlobalConstantBufferUser : register (b2)
  {
    float4 fOcclusionParams  : packoffset(c0);
    
    $ifdef SUPPORTS_SHEARED_MATRIX
      float4x4 mProjectionMatrix : packoffset(c68);
    $else
      float4 fProjectionParams : packoffset(c1);
    $endif
   
    float2   fNoiseOffset : packoffset(c2);
    float    fAmount : packoffset(c3);
    float4   fSpherePoints[64] : packoffset(c4);
  }

#else
  sampler2D          DepthSpecBuffer          : register(s0);
  sampler2D          NormalBuffer             : register(s1);
  sampler2D          NoiseNormalMap           : register(s2);

  float4x4 mModelViewMatrix : register(c22);
  
  float4 fOcclusionParams : register(c32);
  
  $ifdef SUPPORTS_SHEARED_MATRIX
    float4x4 mProjectionMatrix : register(c102);
  $else
    float4 fProjectionParams : register(c33);
  $endif
  
  float2	fNoiseOffset : register(c34);
  float		fAmount : register(c35);
  float4	fSpherePoints[64] : register(c38); 

#endif

struct PS_IN
{                                             
  float4 ProjPos  : SV_Position;              
  float2 UV0 : TEXCOORD1;
  float3 ClipPlanes : TEXCOORD2;
  float3 ViewDirectionEyeSpace : TEXCOORD3;
  float2 UV1 : TEXCOORD4;
};


float4 ps_main( PS_IN In ) : SV_Target
{
  float fDepth = READ_CONVERTED_DEPTH(DepthSpecBuffer, DepthSpecBufferSampler, In.UV0);
  
  float fRangeFactor = fOcclusionParams.x;
  float fOffset = fOcclusionParams.y;
  float fSampleRadius = fOcclusionParams.z;
  
  // Clip AO at the horizon
  clip(0.999f-fDepth);
  
  $ifdef NORMAL_BUFFER_AVAILABLE
    float4 vNormalMat = vTex2D(NormalBuffer, NormalBufferSampler, In.UV0);
	
    // Clip for "always in foreground" material
    clip(vNormalMat.w - 2.0f/255.f);
    float3 vWorldNormal = vNormalMat.xyz;
    vWorldNormal = vWorldNormal * 2.0f - 1.0f;
  $endif
  
  $ifndef SUPPORTS_SHEARED_MATRIX
    float3 vScaledProjParams;
    vScaledProjParams.xy = fProjectionParams.xy * float2(0.5f, -0.5f);
    vScaledProjParams.z = 1.0f;
  $endif
  
  // Sphere center in view space
  float3 vCenterPos = positionForDepth(fDepth, float3(0.0f, 0.0f, 0.0f), In.ViewDirectionEyeSpace);
  
  // Fetch normal at sample center
  $ifdef NORMAL_BUFFER_AVAILABLE
    float3 vRefPlane = mul((float3x3)mModelViewMatrix, vWorldNormal);
	
    #if defined(_VISION_PS3) || defined(_VISION_WIIU)
      // Account for different coordinate system
      vRefPlane.z = -vRefPlane.z;
    #endif
  $else
	// Compute normal by derivative (note that DX9 computes low precision derivatives, so the results may differ between DX9/DX11 - but low precision is still better than no normal at all.)
	float3 centerDx = ddx(vCenterPos);
	float3 centerDy = ddy(vCenterPos);
	
	float3 vRefPlane = normalize(cross(centerDx, centerDy));
	#if defined(_VISION_PS3)
      // Account for different coordinate system (note we flip the whole vector, not only z!)
      vRefPlane = -vRefPlane;
    #endif
	
	// Check that we're not derivating across depth discontinuities by clipping at high slopes
	clip(dot(vRefPlane, -normalize(In.ViewDirectionEyeSpace)) - 0.01f);
  $endif
  
  float fOcclusion = 0.0f;
  
  // Increase sphere radius based on depth to avoid sampling the same fragment over and over
  float fScreenSpaceRadius = fSampleRadius / vCenterPos.z;
  float fScaleRadius = max(0.01f / fScreenSpaceRadius, 1.0f);
  float fScaledSampleRadius = fSampleRadius * fScaleRadius;
  
  float3 vRandomPlane = vTex2D(NoiseNormalMap, NoiseNormalMapSampler, In.UV1+fNoiseOffset).xyz * 2.0f - 1.0f;
  
  for (int i=0; i < $NUM_PASSES; i++)
  {	
  	// Create a randomly rotated sample direction
  	float3 vSampleDirection = reflect(fSpherePoints[i].xyz, vRandomPlane);
  	
  	// Flip sample position to lie in the normal vectors upper hemisphere
  	vSampleDirection *= sign(dot(vSampleDirection, vRefPlane));
  	
  	// Generate view space sample point
  	// The sample direction is in unit length, but to sample the interior as well a length is stored in the w component
  	float3 vSamplePos = vSampleDirection * fScaledSampleRadius * fSpherePoints[i].w + vCenterPos;
  	
  	// Project sample point onto screen position
  	$ifdef SUPPORTS_SHEARED_MATRIX
  		float4 vProjSpherePos;
  		vProjSpherePos = mul(mProjectionMatrix,float4(vSamplePos,1.0f));
  		vProjSpherePos.xy /= vProjSpherePos.w;
  		vProjSpherePos.xy = vProjSpherePos.xy * float2(0.5f, -0.5f) + float2(0.5f, 0.5f);
  	$else
  		float3 vProjSpherePos;
  		vProjSpherePos = vScaledProjParams * vSamplePos;
  		vProjSpherePos.xy /= vProjSpherePos.z;
  		vProjSpherePos.xy += float2(0.5f, 0.5f);
  	$endif
  	
  	// Read depth of sample point
  	float fSampleDepth = READ_CONVERTED_DEPTH(DepthSpecBuffer, DepthSpecBufferSampler, vProjSpherePos.xy);
  	
  	// Check for occlusion
  	float fDepthDiff = vSamplePos.z - fSampleDepth * In.ClipPlanes.y;  // >0 -> occluded!
  	fDepthDiff -= fOffset;
  	
  	// Generate the sample's contribution to the occlusion integral
  	float fWeight = fDepthDiff <= 0 ? 0 : 1;
  	// Modulate sample weight by cosine factor if we can
  	fWeight *= dot(vSampleDirection, vRefPlane);
  		
  	// Divide by expected value of the cosine term so that the resulting weight is comparable to the legacy version without cosine weighting
  	fWeight *= 2;
  	
  	// Attenuate for large depth differences to avoid weighing in samples across depth discontinuities
  	float fAttenuation = 1.0f / (1.0f + fRangeFactor * fDepthDiff * fDepthDiff);
  	
  	fOcclusion += fWeight * fAttenuation;
  }
  
  float fAmbient = ($NUM_PASSES - fOcclusion) / $NUM_PASSES;
  
  fAmbient = lerp(1.0f, fAmbient, fAmount);
  return float4(fAmbient.x, fAmbient.x, fAmbient.x, 1.0f);
}


