#ifdef _VISION_PSP2
  #pragma argument (O3)
#endif

#include <Shaders/VisionCommon.inc>
#include <Shaders/ShaderHelpers.inc>
#include "DeferredShadingHelpers.inc"

#ifdef _VISION_DX11

  $if MSAA_SAMPLES > 1
    #define InputBuffer Texture2DMS<float4, $MSAA_SAMPLES>
    #define SampleBuffer(_tex_, _sampler_, _uv_, _slice_) _tex_.Load(_uv_, _slice_)
  $else
    #define InputBuffer Texture2D<float4>
    #define SampleBuffer(_tex_, _sampler_, _uv_, _slice_) vTex2D(_tex_, _sampler_, _uv_)
  $endif
  
  $if (MSAA_SAMPLES > 1) && !defined (MSAA_SINGLESAMPLE)
    #define SAMPLE_COUNT $MSAA_SAMPLES
  $else
    #define SAMPLE_COUNT 1
  $endif
  

  InputBuffer        DepthSpecBuffer        : register(t0);
  sampler            DepthSpecBufferSampler : register(s0);
  Texture2D <float4> CircleTex        : register(t1);
  sampler            CircleTexSampler : register(s1);
  Texture2D <float4> ShadowTex        : register(t2);
  $if defined(SHADOWING_MODE_PCF4) || defined(SHADOWING_MODE_PCF8) || defined(SHADOWING_MODE_PCF16)
    SamplerComparisonState  ShadowTexSampler : register(s2);
  $else
    sampler            ShadowTexSampler : register(s2);
  $endif
  Texture2D <float4> NoiseTex        : register(t3);
  sampler            NoiseTexSampler : register(s3);
  
  $ifdef CLOUD_MAPPING
    Texture2D <float4> CloudTex        : register(t5);
    sampler            CloudTexSampler : register(s5);
  $endif

  cbuffer g_GlobalConstantBufferUser : register (b2)
  {
    // Common shadow shader constants:
    float3 LightWorldSpace : packoffset(c0);
    float3 LightEyeSpace : packoffset(c1);
    float2 InvShadowMapSize : packoffset(c2);
    float3 LightDirection : packoffset(c3);
    
    float4 CascadeScale : packoffset(c8);
    float4 CascadeTextureSplitsEnd : packoffset(c9);
    float4 FadeoutParameters : packoffset(c10);
    float4x4 LightProjectionTex[4] : packoffset(c12);
    
    // Deferred shadow shader constants:
    float4 ShadowParameters : packoffset(c28);
    float4 InvScreenSize : packoffset(c29);
    float3 UpperLeftCorner : packoffset(c30);
    float3 RightDir : packoffset(c31);
    float3 DownDir : packoffset(c32);
    float3 EyePos : packoffset(c33);
    float4 EyePlane : packoffset(c34);
    
    // Reference cascade index
    int CascadeMaskIndex : packoffset(c35);

    $ifdef CLOUD_MAPPING
      float4 CloudPosition : packoffset(c36);
      float4 CloudExtents  : packoffset(c37);
    $endif
  }

#else
  sampler2D          DepthSpecBuffer : register(s0);
  sampler2D          CircleTex : register(s1);
  sampler2D          ShadowTex : register(s2);
  sampler2D          NoiseTex  : register(s3);
  #ifdef _VISION_PS3
    sampler2D        PatchedShadowTex : register(s4);
  #endif
  $ifdef CLOUD_MAPPING
    sampler2D        CloudTex : register(s5);
  $endif

  // Common shadow shader constants:
  float3 LightWorldSpace : register(c32);
  float3 LightEyeSpace : register(c33);
  float2 InvShadowMapSize : register(c34);
  float3 LightDirection : register(c35);
  
  float4 CascadeScale : register(c36);
  float4 CascadeTextureSplitsEnd : register(c37);
  float4 FadeoutParameters : register(c38);
  float4x4 LightProjectionTex[4] : register(c40);
  
  // Deferred shadow shader constants:
  float4 ShadowParameters : register(c60);
  float4 InvScreenSize : register(c61);
  float3 UpperLeftCorner : register(c62);
  float3 RightDir : register(c63);
  float3 DownDir : register(c64);
  float3 EyePos : register(c65);
  float4 EyePlane : register(c66);

  // Reference cascade index
  float CascadeMaskIndex : register(c67);

  $ifdef CLOUD_MAPPING
    float4 CloudPosition : register(c68);
    float4 CloudExtents  : register(c69);
  $endif
  
#endif





/* 
Supported #defines:
- SHADOWING_CASCADED vs. SHADOWING_NON_CASCADED
- SHADOWING_SPLITMODE_PARALLELSPLIT vs. SHADOWING_SPLITMODE_CONCENTRIC (cascaded only)
- SHADOWING_PROJECTION_ORTHOGRAPHIC vs. SHADOWING_PROJECTION_PERSPECTIVE
- SHADOWING_SIMPLE_FALLBACK_FOR_FARTHER_CASCADES
- SHADOWING_USE_GATHER4
- Debug: SHADOWING_DEBUG_OUTPUT_CASCADECOLORS


*/

#define SHADOWING_NUM_BLOCKERSEARCH_SAMPLES 8

$ifdef SHADOWING_MODE_PCF4
  #define SHADOWING_NUM_SAMPLES 4
$elif defined(SHADOWING_MODE_PCF8)
  #define SHADOWING_NUM_SAMPLES 8
$elif defined(SHADOWING_MODE_PCF16)
  #define SHADOWING_NUM_SAMPLES 16
$elif defined(SHADOWING_MODE_PCF32)
  #define SHADOWING_NUM_SAMPLES 32
$elif defined(SHADOWING_MODE_PCSS16)
  #define SHADOWING_NUM_SAMPLES 16
$elif defined(SHADOWING_MODE_CHS)
  V_REQUIRES_SM41
  #define SHADOWING_NUM_SAMPLES 1
$endif


struct PS_IN                                 
{                                             
  float4 ProjPos  : SV_Position;
  #if defined(_VISION_PS3) || defined(_VISION_PSP2)
    float2 UV0 : WPOS;
  #elif defined(_VISION_DX11)
    float4 UV0 : SV_Position;
  #elif defined(_VISION_XENON)
    float4 UV0 : TEXCOORD0;
  #else
    float2 UV0 : VPOS;
  #endif
    
  #if (SAMPLE_COUNT > 1)
    uint uiSampleIndex : SV_SampleIndex;
  #endif
};

#include <Shaders/ShadowMapHelpers.inc>

int GetCascadeForBoundingBoxSelection(float4 vWorldSpacePos, out float4 vProjCoords, 
  out float fCascadeScale)
{
  // Don't sample the outer 5% of each shadowmap
  const float fBorderBias = 1.05;
	
$if (MAX_CASCADES <= 2)
  const float4 scaleBias[1] = 
  {
    float4(4, 2, -1, -1) * fBorderBias
  };
$elif (MAX_CASCADES == 3 || MAX_CASCADES == 4)
  const float4 scaleBias[3] = 
  {
    float4(4, 4, -1, -1) * fBorderBias,
    float4(4, 4, -3, -1) * fBorderBias,
    float4(4, 4, -1, -3) * fBorderBias
  };
$endif

  for (int i = 0; i < $MAX_CASCADES - 1; i++) 
  {
	// Project worldspace position into texture atlas
    vProjCoords = mul(LightProjectionTex[i], vWorldSpacePos);
	
	// Transform from atlas coordinates so that valid x and y coordinates are in [-1,1] and valid z is < 1
    float3 vPositionInBBox = float3(vProjCoords.xy * scaleBias[i].xy + scaleBias[i].zw, vProjCoords.z);
	
	// Find the outermost coordinate
	vPositionInBBox = abs(vPositionInBBox);
	float fMaxCoord = max(vPositionInBBox.x, max(vPositionInBBox.y, vPositionInBBox.z));
    
	// Check if coords are within cascade bounding box
    if(fMaxCoord < 1)
    {
      fCascadeScale = CascadeScale[i];
      return i;
    }
  }
  
  // Fallback to last cascade
  const int last = $MAX_CASCADES - 1;
  vProjCoords = mul(LightProjectionTex[last], vWorldSpacePos);
  fCascadeScale = CascadeScale[last];
  return last;
}

$if defined(SHADOWING_SPLITMODE_BOUNDINGBOX)
int GetShadowTextureCoordsAndCascadeScale(float4 vWorldSpacePos, out float4 vProjCoords, 
  out float fCascadeScale)
{
  return GetCascadeForBoundingBoxSelection(vWorldSpacePos, vProjCoords, fCascadeScale);
}   
$else
int GetShadowTextureCoordsAndCascadeScale(float4 vWorldSpacePos, out float4 vProjCoords, 
  out float fCascadeScale)
{
  $if defined(SHADOWING_SPLITMODE_INTERVAL)
    int iCascadeIndex = GetCascadeForIntervalSelection(vWorldSpacePos);
  $elif defined(SHADOWING_SPLITMODE_INTERVAL_DEPTH)
	int iCascadeIndex = GetCascadeForIntervalSelectionDepth(vWorldSpacePos);
  $elif defined(SHADOWING_PROJECTION_POINT)
    int iCascadeIndex = GetFaceForPointLight(vWorldSpacePos);
  $else
    int iCascadeIndex = 0;
  $endif

    #if defined(_VISION_PS3) || defined(_VISION_PSP2)

    fCascadeScale = 1.0f;

    // Determine the depth and shadow map coordinates of the pixel
    //saves lots of cmp instructions
    $if (MAX_CASCADES > 1)
      if (iCascadeIndex == 0)
    $endif
    {
      vProjCoords = mul(LightProjectionTex[0], vWorldSpacePos);
      $ifdef SHADOWING_CASCADED
        fCascadeScale = CascadeScale[0];
      $endif
    }
    $if (MAX_CASCADES > 1)
      else 
      $if (MAX_CASCADES > 2)
        if (iCascadeIndex == 1)
      $endif
      {
        vProjCoords = mul(LightProjectionTex[1], vWorldSpacePos);
        $ifdef SHADOWING_CASCADED
          fCascadeScale = CascadeScale[1];
        $endif
      }
      $if (MAX_CASCADES > 2)
        else 
        $if (MAX_CASCADES > 3)
          if (iCascadeIndex == 2)
        $endif
        {
          vProjCoords = mul(LightProjectionTex[2], vWorldSpacePos);
          $ifdef SHADOWING_CASCADED
            fCascadeScale = CascadeScale[2];
          $endif
        }
        $if (MAX_CASCADES > 3)
          else
          { 
            vProjCoords = mul(LightProjectionTex[3], vWorldSpacePos);
            $ifdef SHADOWING_CASCADED
              fCascadeScale = CascadeScale[3];
            $endif
          }
        $endif
      $endif
    $endif

  #else

    $ifdef SHADOWING_CASCADED
      fCascadeScale = CascadeScale[iCascadeIndex];
    $else
      fCascadeScale = 1.0f;
    $endif
  
    // Determine the depth and shadow map coordinates of the pixel
    #ifdef _VISION_DX11
      vProjCoords = mul(LightProjectionTex[iCascadeIndex], vWorldSpacePos);
    #else
      //saves lots of cmp instructions
      $if (MAX_CASCADES > 1)
        if (iCascadeIndex == 0)
      $endif
        vProjCoords = mul(LightProjectionTex[0], vWorldSpacePos);
      $if (MAX_CASCADES > 1)
        else
        $if (MAX_CASCADES > 2)
          if (iCascadeIndex == 1)
        $endif
          vProjCoords = mul(LightProjectionTex[1], vWorldSpacePos);
        $if (MAX_CASCADES > 2)
          else
          $if (MAX_CASCADES > 3)
            if (iCascadeIndex == 2)
          $endif
            vProjCoords = mul(LightProjectionTex[2], vWorldSpacePos);
          $if (MAX_CASCADES > 3)
            else
              vProjCoords = mul(LightProjectionTex[3], vWorldSpacePos);
          $endif
        $endif
      $endif
    #endif
  #endif
  
  $ifndef SHADOWING_CASCADED
    vProjCoords.xyz /= vProjCoords.w;
  $endif

  return iCascadeIndex;
}

$endif

$ifdef CLOUD_MAPPING
float ComputeCloudLuminance(float3 vPos)
{
  // project light vector onto cloud layer (XY mapping)
  float d = (CloudPosition.z - vPos.z) / LightDirection.z;
  float2 uv = (vPos.xy + CloudPosition.xy + d * LightDirection.xy) * CloudExtents.xy;

  // sample luminance value from computed position
  return vTex2D(CloudTex, CloudTexSampler, uv).r;
}
$endif

float2 GetScreenTextureCoords(PS_IN In)
{
  #if defined(_VISION_DX11) || defined(_VISION_WIIU)
    In.UV0.xy = In.UV0.xy * InvScreenSize.xy;
  #elif defined(_VISION_XENON)
    In.UV0.xy = (In.UV0.xy / In.UV0.w) * 0.5 + 0.5;    
    In.UV0.xy += 0.5f * InvScreenSize.xy;
  #elif defined(_VISION_PS3)
    In.UV0.xy = In.UV0.xy * InvScreenSize.xy;
    In.UV0.y = 1.0f - In.UV0.y;
  #elif defined(_VISION_PSP2)
    In.UV0.xy = In.UV0.xy * InvScreenSize.xy;
  #else
    In.UV0.xy = In.UV0.xy * InvScreenSize.xy + 0.5f * InvScreenSize.xy;
  #endif

  return In.UV0.xy;
}

float4 ps_main( PS_IN In ) : SV_Target
{
  In.UV0.xy = GetScreenTextureCoords(In);
  
  // Determine world-space position
  $if MSAA_SAMPLES > 1

    #if (SAMPLE_COUNT == 1)
      int iSample = 0;
    #else
      int iSample = In.uiSampleIndex;
    #endif
    
    int2 uv = int2(In.ProjPos.xy);
    float fDepth = DepthSpecBuffer.Load(uv.xy, iSample).r;
  $else
    float fDepth = READ_CONVERTED_DEPTH(DepthSpecBuffer, DepthSpecBufferSampler, In.UV0.xy);
  $endif
    
  #if defined(_VISION_PSP2)
    clip(0.999f - fDepth);
  #else
    clip(0.99999f - fDepth);
  #endif

  
  float3 ViewDirection = UpperLeftCorner + RightDir * In.UV0.x + DownDir * In.UV0.y;
  float3 vPos = positionForDepth(fDepth, EyePos, ViewDirection);
  float4 pos4 = float4(vPos, 1.0f);

  // Determine cascade - multiple cascades are only used for directional or point lights
  float fCascadeScale = 1.0;
  float4 vProjCoords = 0.0;
  int iCascadeIndex = GetShadowTextureCoordsAndCascadeScale(pos4, vProjCoords, fCascadeScale);
  float fRadius = ShadowParameters.x + fDepth * ShadowParameters.y;

  $ifdef SHADOWING_LAYERMASK
    float test = CascadeMaskIndex == iCascadeIndex;
    clip(test - 0.5);
    return float4(0,0,0,1);
  $endif

  float shadowTerm;

  // Compute the final shadowing term using the desired filtering method
  $ifdef SHADOWING_SIMPLE_FALLBACK_FOR_FARTHER_CASCADES
		if (iCascadeIndex >= 1)
		{
			shadowTerm = ComputeShadowTerm_PCF4(vProjCoords);
		}
		else
  $endif
    {
  $ifdef SHADOWING_MODE_PCF4
    shadowTerm = ComputeShadowTerm_PCF4(vProjCoords);
  $elif defined(SHADOWING_MODE_PCF8) || defined(SHADOWING_MODE_PCF16) || defined(SHADOWING_MODE_PCF32)
    shadowTerm = ComputeShadowTerm_PCFN(vProjCoords, fCascadeScale, fRadius, In.UV0.xy);
  $elif defined(SHADOWING_MODE_PCSS16)
    shadowTerm = ComputeShadowTerm_PCSS(vProjCoords, fCascadeScale, fDepth, In.UV0.xy);
  $elif defined(SHADOWING_MODE_CHS)
    shadowTerm = ComputeShadowTerm_CHS(vProjCoords, fCascadeScale, fRadius);
  $endif
    }
    
  $ifdef CLOUD_MAPPING
    shadowTerm *= ComputeCloudLuminance(vPos);
  $endif
    
  $ifdef SHADOWING_CASCADED
    // TODO: Alternative simpler version?
    float fFadeOut = 1.0f - saturate((FadeoutParameters.x - length(vPos.xyz - EyePos)) * FadeoutParameters.y);
    shadowTerm = saturate(shadowTerm + fFadeOut);
  $endif

  $ifdef SHADOWING_DEBUG_OUTPUT
    // the ternary operator crashes the NGP shader compiler.
    //return (shadowTerm * 0.5 + 0.5) * float4((iCascadeIndex==1||iCascadeIndex==3)?1.0f:0.0f, (iCascadeIndex==0||iCascadeIndex==1)?1.0f:0.0f, iCascadeIndex==2?1.0f:0.0f, 1.0f);
    float4 tint = float4(0,0,0,1);
    if (iCascadeIndex == 1 || iCascadeIndex == 3) { tint.x = 1; }
    if (iCascadeIndex < 3) { tint.y = 1; }
    if (iCascadeIndex == 2) { tint.z = 1; }
  
    return (shadowTerm * 0.5 + 0.5) * tint;
  $else
    return shadowTerm;
  $endif
}


