#include <Shaders/VisionCommon.inc>
#include "ShaderHelpers.inc"

#if defined(VMATERIAL_BUMPCUBEMAP) && !defined(VMATERIAL_BUMP)
  #define VMATERIAL_BUMP
#endif

#if defined(VMATERIAL_BUMPCUBEMAP) && !defined(VMATERIAL_CUBEMAP)
  #define VMATERIAL_CUBEMAP
#endif


#ifdef _VISION_DX11

  cbuffer g_GlobalConstantBufferFrame : register (b0)
  {
    float4    Timer           : packoffset(c16);
    float4    eyePos          : packoffset(c17);  // eye position
    float4    depthFog        : packoffset(c19);  // linear depth fog parameter: near, far, 1/(far-near)
  }

  cbuffer g_GlobalConstantBufferObject : register (b1)
  {
    float4x4  matMV   : packoffset(c0);   // model view matrix
    float4x4  matMVP  : packoffset(c4);   // model view projection matrix
    float4x4  matMtoW : packoffset(c8);   // model to worldspace matrix
    float4    MUVtoLM : packoffset(c13);  // model UV to lightmap
  }

  cbuffer g_GlobalConstantBufferUser : register (b2)
  {
    float4 LightPos : packoffset(c0);
    #if !defined (LIGHTGRID) && defined (VMATERIAL_BUMP)
      float4 LightDir : packoffset(c1);
    #endif
    float4 ScrollSpeed : packoffset(c2);
  }
  cbuffer g_GlobalConstantBufferLightGrid : register (b3)
  {
    #ifdef LIGHTGRID
      float4 LightDir : register(c2);
    #endif
  }      
#else
  float4x4 matMVP : register(c8);
  float4x4 matMtoW : register(c20);
  float4 eyePos : register(c31);

  #ifdef LIGHTGRID
    float4 LightDir : register(c34);
  #elif defined(VMATERIAL_BUMP)
    float4 LightDir : register(c64);
  #endif

  #ifndef LIGHTGRID
    float4 MUVtoLM : register(c60);
  #endif
  #ifdef USE_FOG
    float4 depthFog : register(c62);  // linear depth fog parameter: near, far, 1/(far-near)
  #endif

  #if defined(VMATERIAL_BUMPANIMATED)
    float4 Timer : register(c30);
    float4 ScrollSpeed : register(c65);
  #endif
#endif


struct VS_IN
{
  float3 ObjPos   : POSITION;

  #if defined(VMATERIAL_BUMP) || defined(VMATERIAL_CUBEMAP) || defined(LIGHTGRID)
    float3 Normal : NORMAL;
  #endif
  
  float2 UV0 : TEXCOORD0;
  float2 UV1 : TEXCOORD1;
  
  #if defined(VMATERIAL_BUMP)
    float3 Tangent : TEXCOORD2;
  #endif
};

struct VS_OUT
{
  float4 ProjPos  : SV_Position;
  half4 UV0 : TEXCOORD0;  //diffuse & normalmap
  half4 UV1 : TEXCOORD1;  //lightmap

  #ifdef USE_FOG
    half fogCoord : TEXCOORD2;
  #endif
  #if defined(VMATERIAL_BUMP) || defined(LIGHTGRID)
    half3 LightDir : TEXCOORD3;
  #endif
  #if (defined(LIGHTGRID) || defined(VMATERIAL_CUBEMAP)) && (!defined(VMATERIAL_BUMP) || defined(VMATERIAL_BUMPCUBEMAP))
    half3 Normal : TEXCOORD5;
  #endif
  #if defined(VMATERIAL_CUBEMAP)
    half3 EyeDir : TEXCOORD4;  
  #endif
  #if defined(VMATERIAL_BUMPCUBEMAP)  
    half3 Tangent : TEXCOORD6;
    half3 BiNormal : TEXCOORD7;
  #endif
};


VS_OUT vs_main(VS_IN In)
{
  VS_OUT Out;
  float4 ObjPos = float4(In.ObjPos, 1.0f);
  Out.ProjPos = mul(matMVP, ObjPos);

  Out.UV0 = half4(In.UV0, 0, 0); // base texture
  
  #ifndef LIGHTGRID
    Out.UV1 = half4(In.UV1 * MUVtoLM.xy + MUVtoLM.zw, 0, 0); // lightmap
  #else
    Out.UV1 = Out.UV0;
  #endif

  #if defined(VMATERIAL_BUMP)
    half3x3 matMtoWRot = (half3x3)matMtoW;
    half3 Normal = mul(matMtoWRot, In.Normal);
    Normal = normalize(Normal);
    half3 Tangent = mul(matMtoWRot, In.Tangent);
    Tangent = normalize(Tangent);
    half fMul = dot(In.Tangent, In.Tangent) - 2.0h;  
    half3 BiNormal = cross(Tangent, Normal)*fMul;
    BiNormal = normalize(BiNormal);

    #if defined(VMATERIAL_BUMPCUBEMAP) 
      float4 worldPos = mul(matMtoW, ObjPos);
      Out.EyeDir = worldPos.xyz - eyePos.xyz;
      Out.Tangent = Tangent;
      Out.BiNormal = -BiNormal;
      Out.Normal = Normal;

      #if defined(VMATERIAL_BUMPANIMATED)
        Out.UV1 = Out.UV1.xyxy + Timer.xxxx * ScrollSpeed;
      #endif
    #endif

    Out.LightDir.x = dot(Tangent, LightDir.xyz);
    Out.LightDir.y = -dot(BiNormal, LightDir.xyz);
    Out.LightDir.z = dot(Normal, LightDir.xyz);
  #elif defined(LIGHTGRID)
    Out.LightDir = LightDir.xyz;
    half3x3 matMtoWRot = (half3x3)matMtoW;
    half3 Normal = mul(matMtoWRot, In.Normal);
    Normal = normalize(Normal);
    Out.Normal = Normal;
  #endif

  #if defined(VMATERIAL_CUBEMAP) && !defined(VMATERIAL_BUMPCUBEMAP)
    Out.Normal = mul((float3x3)matMtoW, In.Normal);
    Out.Normal = normalize(Out.Normal);
    float4 worldPos = mul(matMtoW, ObjPos);
    Out.EyeDir = worldPos.xyz - eyePos.xyz;
  #endif

  #ifdef USE_FOG
    Out.fogCoord = GetFogCoord(depthFog, Out.ProjPos.z);
  #endif

  return Out;
}