//
#include <Shaders/VisionCommon.inc>

#ifdef _VISION_DX11
  cbuffer g_GlobalConstantBufferFrame : register (b0)
  {
    float4x4 matInvEye     : packoffset(c8);  // inverse eye matrix
  }
  cbuffer g_GlobalConstantBufferObject : register (b1)
  {
    float4x4 matMVP        : packoffset(c4); // model view projection matrix
  }

  #ifdef _IS_VERTEX_SHADER
    cbuffer g_GlobalConstantBufferUserVS : register (b2)
    {
      float4   vStartPos     : packoffset(c0);
      float4   vEndPos       : packoffset(c1);
      float    fParamScale   : packoffset(c2);
      #ifdef USE_SHADOW
        float4   ConePlanes[4] : packoffset(c3); // 0-2 cone projection planes , 3:clip plane params; same as c24, but that one is not current
      #endif
    }
  #endif  //_IS_VERTEX_SHADER

  #ifdef _IS_PIXEL_SHADER
    cbuffer g_GlobalConstantBufferUserPS : register (b2)
    {
      float4   vConeColor    : packoffset(c0);
    }
  #endif  //_IS_PIXEL_SHADER

#else
  float4x4 matMVP        : register(c8);
  float4x4 matInvEye     : register(c16);

  // volumetric cone specific (vs)
  #if defined(_VISION_PS3) || defined(_VISION_PSP2) || defined(_VISION_GLES2) || defined(_VISION_WIIU)
    float4   vStartPos     : register(c64);
    float4   vEndPos       : register(c65);
    float    fParamScale   : register(c66);
    float4   ConePlanes    : register(c67); // 0-2 cone projection planes , 3:clip plane params; same as c24, but that one is not current
    float4   ConePlanes1   : register(c68);
    float4   ConePlanes2   : register(c69);
    float4   ConePlanes3   : register(c70);
  #else
    float4   vStartPos;
    float4   vEndPos;
    float    fParamScale;
    float4   ConePlanes[4]; // 0-2 cone projection planes , 3:clip plane params; same as c24, but that one is not current
  #endif
  
  // ps
  #if defined(_VISION_PS3) || defined(_VISION_PSP2) || defined(_VISION_GLES2) || defined(_VISION_WIIU)
    float4   vConeColor    : register(c32);
  #else
    float4   vConeColor;
  #endif

#endif

struct VS_IN
{
  float3 ObjPos  : POSITION;
};

struct VS_OUT
{
  float4 ProjPos : SV_Position;
  float2 UV0     : TEXCOORD0;
  #ifdef USE_SHADOW
    float4 ProjCoord : TEXCOORD1;
  #endif
};

#define PS_IN VS_OUT

#ifdef _IS_VERTEX_SHADER

  ////////////////////////////////////////////////////////////////////////////////////////////
  // Vertex shader
  ////////////////////////////////////////////////////////////////////////////////////////////

  float4 ComputeStretchedPos(float3 inPos)
  {
#if defined(_VISION_WIIU)
    float4 camRight = matInvEye._m00_m01_m02_m03;
    float4 camUp = matInvEye._m10_m11_m12_m13;
#else
    // The following two lines are correct for PC and PS3, since they implicitly already take row-major/column-major ordering into account
    float4 camRight = matInvEye._m00_m10_m20_m30;
    float4 camUp = matInvEye._m01_m11_m21_m31;
#endif

    float fParam = fParamScale*inPos.z; // [0..1]
    float4 center = lerp(vStartPos,vEndPos,fParam);
    float2 span = (inPos-0.5)*center.w;
    float4 absPos = center + camRight*span.x + camUp*span.y;
    absPos.w = 1.0;
    return absPos;
  }

  VS_OUT vs_main( VS_IN In )
  {
    VS_OUT Out;
    float4 absPos = ComputeStretchedPos(In.ObjPos);
    Out.ProjPos = mul( matMVP, absPos);
    Out.UV0 = In.ObjPos.xy;
    
    #ifdef USE_SHADOW
      #if defined(_VISION_PS3) || defined(_VISION_PSP2) || defined(_VISION_GLES2) || defined(_VISION_WIIU)
        Out.ProjCoord.x = 0.5*dot(ConePlanes, absPos);
        Out.ProjCoord.y = 0.5*dot(ConePlanes1, absPos);
        Out.ProjCoord.w = dot(ConePlanes2, absPos); // this is also the distance to lightsource
        Out.ProjCoord.z = (Out.ProjCoord.w-ConePlanes3.x)*ConePlanes3.z; // mapped to [0..1] range using cone context clip planes
      #else   
        Out.ProjCoord.x = 0.5*dot(ConePlanes[0], absPos);
        Out.ProjCoord.y = 0.5*dot(ConePlanes[1], absPos);
        Out.ProjCoord.w = dot(ConePlanes[2], absPos); // this is also the distance to lightsource
        Out.ProjCoord.z = (Out.ProjCoord.w-ConePlanes[3].x)*ConePlanes[3].z; // mapped to [0..1] range using cone context clip planes
      #endif   
    #endif   
   
    return Out;
  }

#endif //_IS_VERTEX_SHADER


#ifdef _IS_PIXEL_SHADER

////////////////////////////////////////////////////////////////////////////////////////////
// Pixel shader
////////////////////////////////////////////////////////////////////////////////////////////


  #ifdef _VISION_DX11
    Texture2D BaseTexture        : register(t0);
    sampler   BaseTextureSampler : register(s0);
    Texture2D RenderTargetTex        : register(t1);
    sampler   RenderTargetTexSampler : register(s1);
  #else
    sampler2D BaseTexture        : register(s0);
    sampler2D RenderTargetTex        : register(s1);
  #endif


  float4 ps_main( PS_IN In ) : SV_Target
  {
    #ifdef USE_SHADOW
      float2 projTexCoord = 0.5f - In.ProjCoord.xy / In.ProjCoord.w;
      float dist = vTex2D1Comp(RenderTargetTex, RenderTargetTexSampler, projTexCoord).x;
      clip(dist-In.ProjCoord.z); // depth test
    #endif

    float4 tc = vTex2D(BaseTexture, BaseTextureSampler, In.UV0);
    return tc * vConeColor;
  }

#endif
