// Includes
#include <Shaders/VisionCommon.inc>
#include <Shaders/DeferredShadingHelpers.inc>
#include <Shaders/FogHelpers.inc>

/////////////////////////////////////////////////////////////////////////////
// Pixel shader sources
/////////////////////////////////////////////////////////////////////////////
#if defined(_VISION_DX11)
  Texture2D <float4> BaseTexture        : register(t0);
  sampler            BaseTextureSampler : register(s0);
  Texture2D <float4> DepthTexture        : register(t1);
  sampler            DepthTextureSampler : register(s1);
  Texture2D <float4> CausticsTexture        : register(t2);
  sampler            CausticsTextureSampler : register(s2);
  Texture2D <float4> NormalMap        : register(t3);
  sampler            NormalMapSampler : register(s3);
  Texture2D <float4> FoamTexture        : register(t4);
  sampler            FoamTextureSampler : register(s4);
  Texture2D <float4> SceneTexture        : register(t5);
  sampler            SceneTextureSampler : register(s5);


#else
  sampler2D          BaseTexture        : register(s0);
  sampler2D          DepthTexture        : register(s1);
  sampler2D          CausticsTexture        : register(s2);
  sampler2D          NormalMap        : register(s3);
  sampler2D          FoamTexture        : register(s4);
  sampler2D          SceneTexture        : register(s5);
#endif

#if defined(_VISION_DX11)

cbuffer g_GlobalConstantBufferFramePS : register (b0)
{
  float4 Timer               : packoffset(c16);
  
#ifdef USE_FOG  
  float4 DepthFogCol         : packoffset(c20);  // linear depth fog color
  float4 LayeredFogParams    : packoffset(c26);  // x = HeightFalloff, y = GroundOFfset, zw = FogExtents
  float4 LayeredColorDensity : packoffset(c27);  // xyz = BaseColor, w = Density  
#endif

  float4 CameraPos           : packoffset(c17);
  #ifdef USE_TIME_OF_DAY
    float4 g_AmbientColor : packoffset(c21);
  #endif
  
}

cbuffer g_GlobalConstantBufferModelPS : register (b1)
{
  float4x4 ModelViewMatrix_PS : packoffset(c0);
}

cbuffer g_GlobalConstantBufferUserPS : register (b2)
{
  // General Shader Parameters
  float3 UpperLeftCorner : packoffset(c2);
  float3 RightDir : packoffset(c3);
  float3 DownDir : packoffset(c4);
  float2 InvScreenSize : packoffset(c5);
  float4 ClipPlanes : packoffset(c6);
  
  // Shore Parameters
  float fShoreFadeoutRange : packoffset(c7);
  
  // General Water Parameters
  float3 waterNormalParams : packoffset(c8);
  float2 waterFlowDirection : packoffset(c9);
  float  waterOpacity : packoffset(c10);
  float3 waterTintColor : packoffset(c11);
  
  // Sun Parameters
  float3 sunDirection : packoffset(c12);
  float3 sunColor : packoffset(c13);
  
  // Reflection
  float4 reflectionParams : packoffset(c14);

  // Caustics
  float4 causticsParams : packoffset(c15);

  // Refraction  
  float refractionDepthScale : packoffset(c16);

  // Foam
  float3 foamWobble : packoffset(c17);
  float4 foamFadeParams : packoffset(c18);
  
  // Fog
  float3 brightWaterColor : packoffset(c19);
  float3 darkWaterColor : packoffset(c20);
  float3 chromaticExtinction : packoffset(c21);

  // Specular
  float2 specularParams : packoffset(c22);    

#ifdef WATER_CAUSTICS_ROTATION
   float4 causticsScrollTransform : packoffset(c23);
#endif
}  
#elif defined(_VISION_PS3) || defined(_VISION_PSP2) || defined(_VISION_GLES2) || defined(_VISION_WIIU)
  float4 DepthFogCol : register(c1);            // linear depth fog color
  float4 LayeredFogParams : register(c16);      // layered fog parameters
  float4 LayeredColorDensity : register(c17);   // layered fog color/density
  float4 CameraPos : register(c19);
  
  // General Shader Parameters
  float4 Timer : register(c18);
  
  #ifdef USE_TIME_OF_DAY
    float4 g_AmbientColor : register(c21);
  #endif
  
  float4x4 ModelViewMatrix_PS : register(c22);

  float4 ReflectionColor : register(c33);
  float3 UpperLeftCorner : register(c34);
  float3 RightDir : register(c35);
  float3 DownDir : register(c36);
  float2 InvScreenSize : register(c37);
  float4 ClipPlanes : register(c38);
  
  // Shore Parameters
  float fShoreFadeoutRange : register(c39);
  
  // General Water Parameters : register(c40);
  float3 waterNormalParams : register(c41);
  float2 waterFlowDirection : register(c42);
  float  waterOpacity : register(c43);
  float3 waterTintColor : register(c44);
  
  // Sun Parameters
  float3 sunDirection : register(c45);
  float3 sunColor : register(c46);
  
  // Reflection
  float4 reflectionParams : register(c47);

  // Caustics
  float4 causticsParams : register(c48);
 
  // Refraction  
  float refractionDepthScale : register(c49);

  // Foam
  float3 foamWobble : register(c50);
  float4 foamFadeParams : register(c51);
  
  // Fog
  float3 brightWaterColor : register(c52);
  float3 darkWaterColor : register(c53);
  float3 chromaticExtinction : register(c54);

  // Specular
  float2 specularParams : register(c55);
#ifdef WATER_CAUSTICS_ROTATION
  float4 causticsScrollTransform  : register(c56);
#endif
#else

  float4 DepthFogCol : register(c1);            // linear depth fog color
  float4 LayeredFogParams : register(c16);      // layered fog parameters
  float4 LayeredColorDensity : register(c17);   // layered fog color/density
  float4 CameraPos : register(c19);
  
  float4 Timer : register(c18);
  
  #ifdef USE_TIME_OF_DAY
    float4 g_AmbientColor : register(c21);
  #endif

  float4x4 ModelViewMatrix_PS : register(c22);
  
  // General Shader Parameters
  float4 ReflectionColor;
  float3 UpperLeftCorner;
  float3 RightDir;
  float3 DownDir;
  float2 InvScreenSize;
  float4 ClipPlanes;
  
  // Shore Parameters
  float fShoreFadeoutRange;
  
  // General Water Parameters
  float3 waterNormalParams;
  float2 waterFlowDirection;
  float  waterOpacity;
  float3 waterTintColor;
  
  // Sun Parameters
  float3 sunDirection;
  float3 sunColor;
  
  // Reflection
  float4 reflectionParams;

  // Caustics
  float4 causticsParams;
#ifdef WATER_CAUSTICS_ROTATION
  float4 causticsScrollTransform;
#endif  
  // Refraction  
  float refractionDepthScale;

  // Foam
  float3 foamWobble;
  float4 foamFadeParams;
  
  // Fog
  float3 brightWaterColor;
  float3 darkWaterColor;
  float3 chromaticExtinction;

  // Specular
  float2 specularParams;  
  
#endif

struct PS_IN
{                  
  float4 ProjPos : SV_Position;  
  float4 UVProj  : TEXCOORD0; 
  float4 EyePosAndDepth : TEXCOORD1;
  float2 UVBase : TEXCOORD2;
  
  #ifdef WATER_SM20  
  float4 WorldSpacePos : TEXCOORD3;
  #else
  float3 WorldSpacePos : TEXCOORD3;
  #endif
  
  #ifdef USE_FOG
    float2 fogCoord : TEXCOORD4;
  #endif
  
  #ifdef USE_TIME_OF_DAY
    float3 vSunDirection : TEXCOORD5;
    float3 vSunColor : TEXCOORD6;
  #endif
    
  #if defined(_VISION_PS3) || defined(_VISION_PSP2)
    float4 ScreenPos : WPOS;
  #elif defined(_VISION_DX11) || defined(_VISION_DX9) || defined(_VISION_XENON)
    float4 ScreenPos : TEXCOORD7;
  #elif defined(_VISION_DX11)
    float4 ScreenPos : SV_Position;
  #else
    float4 ScreenPos : VPOS;
  #endif
  
};


#include "ShaderHelpers.inc"
#include "WaterShaderHelpers.inc"


float4 ps_main( PS_IN In ) : SV_Target
{
  float2 screenPos = _GetScreenTexPos(In.ScreenPos, InvScreenSize);
  float3 ViewDirection = UpperLeftCorner + RightDir * screenPos.x + DownDir * screenPos.y;
  float fWaterLevel = In.WorldSpacePos.z;
  #ifdef WATER_SM20
    float fTime = In.WorldSpacePos.w;
  #else
    float fTime = Timer.x;
  #endif
  
  #ifdef USE_TIME_OF_DAY
    float3 vSunDirection = In.vSunDirection;
    float3 vSunColor = sunColor; //normalize(In.vSunColor);
  #else
    float3 vSunDirection = sunDirection;
    float3 vSunColor = sunColor; 
    vSunDirection.z *= -1.f;
  #endif
    
  float2 scaledUvBase = In.UVBase * waterNormalParams.y;
  float4 vNormalAndHeight = GetWaterNormalAndHeightFromNormalMap(scaledUvBase, fTime * waterNormalParams.x, fTime * waterFlowDirection, waterNormalParams.z, NormalMap, NormalMapSampler);
  float3 vNormalizedViewDir = normalize(ViewDirection);
  
  #ifdef HAS_DEPTH_INFORMATION
    float fDepth = READ_CONVERTED_DEPTH(DepthTexture, DepthTextureSampler, screenPos).x;
    float3 vPos = positionForDepth(fDepth, In.EyePosAndDepth.xyz, ViewDirection);
   
    float fWaterDepth = fWaterLevel - vPos.z;
    float fShoreFade = saturate(fWaterDepth * fShoreFadeoutRange);
  #else
    float fShoreFade = 1.0f;
    float fWaterDepth = 100.f;
  #endif
  
    
  #ifdef WATER_REFLECTION        
    float3 vReflColor = GetWaterReflection(vNormalAndHeight.xyz, reflectionParams.x, In.UVProj, BaseTexture, BaseTextureSampler);
    float fresnelTerm = GetFresnelTerm(vNormalAndHeight.xyz, vNormalizedViewDir, reflectionParams.y) * reflectionParams.z + reflectionParams.w;
    
    // Shore fadeout
    float4 cReflectionColor = float4(vReflColor, saturate(fresnelTerm*fShoreFade));
  #else
    float4 cReflectionColor = float4(0.0f, 0.0f, 0.0f, 0.0f);
  #endif
  
  #ifdef WATER_REFRACTION   
    // "500.f" is a tweak value which was necessary to ensure backwards compatibility with previous versions.
    #ifdef HAS_DEPTH_INFORMATION
      float  refractionNormalScale = (InvScreenSize * fWaterDepth * 500.f * refractionDepthScale) / (fDepth * ClipPlanes.y);
    #else
      float  refractionNormalScale = (InvScreenSize * fWaterDepth * refractionDepthScale);
    #endif
    float2 refractionCoords;
    float3 cRefrColor = GetWaterRefraction(vNormalAndHeight.xyz, refractionNormalScale, screenPos, SceneTexture, SceneTextureSampler, refractionCoords);

    #ifdef HAS_DEPTH_INFORMATION
      fDepth = READ_CONVERTED_DEPTH(DepthTexture, DepthTextureSampler, refractionCoords).x;
      float3 vPosPostRefraction = positionForDepth(fDepth, In.EyePosAndDepth.xyz, ViewDirection);
      float fWaterDepthPostRefraction = fWaterLevel - vPosPostRefraction.z;
    #endif
  #else
    #ifdef HAS_DEPTH_INFORMATION
      float3 vPosPostRefraction = vPos;
      float fWaterDepthPostRefraction = fWaterDepth;    
    #endif
    float3 cRefrColor = float3(0.0f, 0.0f, 0.0f);
  #endif
  
  #if defined(WATER_CAUSTICS) && defined(HAS_DEPTH_INFORMATION)
   
  #ifdef WATER_CAUSTICS_ROTATION 
    float4 transform = causticsScrollTransform; // this 2x2 matrix is tracked by the cpp code
  #else
    float4 transform = float4(1,0,0,1); // identity matrix
  #endif
    
    float3 vNormal = GetCausticsNormalFromNormalMap(vPosPostRefraction.xy * causticsParams.x, fTime * causticsParams.y, fTime * waterFlowDirection, causticsParams.z, transform, NormalMap, NormalMapSampler); 
    float fFadedCausticsIntensity = causticsParams.w * (1.0f - saturate(fWaterDepthPostRefraction * waterOpacity));
    
    #ifdef CAUSTICS_CHROMATIC_DISPERSION
      float3 cCaustics = GetCausticsFromMapWithChromaticDispersion(vNormal, float3(0.0f, 0.0f, 1.0f), CausticsTexture, CausticsTextureSampler);
    #else
      float3 cCaustics = GetCausticsFromMap(vNormal, float3(0.0f, 0.0f, 1.0f), CausticsTexture, CausticsTextureSampler);
    #endif
    cCaustics *= fFadedCausticsIntensity;
  #else
    float3 cCaustics = float3(0.0f, 0.0f, 0.0f);
  #endif
  
  #if defined(WATER_FOAM) && defined(HAS_DEPTH_INFORMATION)
    float3 cFoamColor = GetWaterFoam(In.UVBase * foamWobble.z, waterFlowDirection, foamWobble.x, foamWobble.y, vPos.xyz, fTime, FoamTexture, FoamTextureSampler);
    cFoamColor = GetFoamFade(cFoamColor.xyz, fWaterDepth, foamFadeParams.xy, vNormalAndHeight.w, foamFadeParams.z, foamFadeParams.w);
    cFoamColor *= fShoreFade;
  #else
    float3 cFoamColor = float3(0.0f, 0.0f, 0.0f);
  #endif
  
  //float3 cFloorColor = cRefrColor * saturate(cCaustics + 0.65f + (1.0f - fShoreFade));
  //float3 cFloorColor = cRefrColor * (0.85f + cCaustics * fShoreFade);
  #ifdef CAUSTICS_CHROMATIC_DISPERSION
    float3 cFloorColor = cRefrColor * lerp(float3(1.0f, 1.0f, 1.0f), 0.85f + cCaustics, fShoreFade);
  #else
    float3 cFloorColor = cRefrColor * lerp(1.0f, 0.85f + cCaustics.r, fShoreFade);
  #endif
  
  #if defined(WATER_FOG) && defined(HAS_DEPTH_INFORMATION)   
    float fDensityViewDir = saturate((fDepth*ClipPlanes.y - In.EyePosAndDepth.w) * waterOpacity);    
    float fDensityDepth = saturate(fWaterDepthPostRefraction * waterOpacity);
    float3 fChromaticExtinction = saturate(fDensityDepth*chromaticExtinction);
    
    float3 waterColorViewDir = lerp(cFloorColor, brightWaterColor, fDensityViewDir);
    float3 cFogColor = lerp(waterColorViewDir, darkWaterColor, fChromaticExtinction);
  #else
    float3 cFogColor = cFloorColor * waterTintColor;
  #endif

  #ifdef WATER_SPECULARITY
    float3 vHalf = normalize(vNormalizedViewDir + vSunDirection);
    float dotSpec = saturate(-dot(vHalf, vNormalAndHeight.xyz));
    float3 cSpecularColor = specularParams.y * pow(dotSpec, specularParams.x) * vSunColor;
  #else
    float3 cSpecularColor = float3(0.0f, 0.0f, 0.0f);
  #endif
    
  float4 cResultColor;
  cResultColor.xyz = lerp(cFogColor, cReflectionColor.xyz, cReflectionColor.a);
  cResultColor.xyz += cSpecularColor * fShoreFade;

  #ifdef USE_TIME_OF_DAY
    cResultColor.xyz = lerp(cResultColor.xyz, max(In.vSunColor, g_AmbientColor), saturate(cFoamColor.x*2.0f));
  #else
    cResultColor.xyz = lerp(cResultColor.xyz, vSunColor, saturate(cFoamColor.x*2.0f));
  #endif
  cResultColor.w = 1.0f;
  
  #ifdef USE_FOG
  
    #if defined(WATER_NO_LAYERED_FOG)
      cResultColor.xyz = GetFogColor(cResultColor.xyz, DepthFogCol.xyz, (In.ScreenPos.z-In.fogCoord.x) / (In.fogCoord.y - In.fogCoord.x));
    #else
      float4 depthParams = float4(In.fogCoord.xy, 1.0f / (In.fogCoord.y - In.fogCoord.x), 0);
      
      float scrDepth = (In.ScreenPos.z - ClipPlanes.x) * ClipPlanes.z;

      float4 A, B;
      A.xyz = DepthFogCol.xyz;
      A.w = min(DepthFogCol.w, saturate(ComputeDepthFogOpacity(scrDepth, depthParams, ClipPlanes)));

      B = float4(LayeredColorDensity.xyz, 0);

      if (LayeredColorDensity.w > 0)
      {
        B.w = saturate(ComputeLayeredFogOpacity(scrDepth * ViewDirection, LayeredFogParams, LayeredColorDensity.w, ClipPlanes, CameraPos));
      }
      
      float4 fogColor = MixFog(A, B);

      cResultColor.xyz = cResultColor.xyz * fogColor.w + fogColor.xyz;
    #endif
  #endif
  
  return cResultColor;
}
