#include <Shaders/VisionCommon.inc>
#include <Shaders/ShaderHelpers.inc>
#include "DeferredShadingHelpers.inc"

#ifdef _VISION_DX11
  Texture2D <float4> DepthSpecBuffer        : register(t0);
  sampler            DepthSpecBufferSampler : register(s0);
  Texture2D <float4> NoiseNormalMap         : register(t2);
  sampler            NoiseNormalMapSampler  : register(s2);

  cbuffer g_GlobalConstantBufferUser : register (b2)
  {
    float4 fOcclusionParams  : packoffset(c0);
    float4 fThreshold : packoffset(c1);
    float  fAmount : packoffset(c2);
  }

#else
  sampler2D          DepthSpecBuffer          : register(s0);
  sampler2D          NoiseNormalMap           : register(s2);

  float4 fOcclusionParams : register(c32);
  float4 fThreshold : register(c33);
  float  fAmount : register(c34);
#endif

struct PS_IN
{                                             
  float4 ProjPos  : SV_Position;              
  float3 UV0 : TEXCOORD1;
  float3 ClipPlanes : TEXCOORD2;
  float3 ViewDirectionEyeSpace : TEXCOORD3;
  float2 UV1 : TEXCOORD4;
};

float CalculateRadius(float radii, float depth, float2x2 rotation, float aspectRatio, float2 UV0, float g_FarPlaneZ, float maxDistance)
{
	float occlusionAmount = 0.0f;
	
	float fDepth = depth;
	depth *= g_FarPlaneZ;

	#define NUM_PAIRS 3
	float2 sphereSamplePoints[3] = { {-0.002801121, -0.5014006},{0.6414566, -0.3781513},{0.2156863, 0.1204482} };
	float3 sphereWidths = float3(1.730421, 1.334968, 1.938011);
	float3 sphereWeights = float3(0.182984, 0.1531844, 0.1390501);
	
	// The radius should get smaller the further away it is
	float radius = radii / depth;
	// Don't let the radius get too big (for performance reasons)
	//radius = min( radius, 0.07f );
	// Take samples!
	for ( int j = 0; j < NUM_PAIRS; j++ )
	{
		// Each pair of sample points is made by reflection
		float2 samplePoint = sphereSamplePoints[j];
		// Rotate it (based on the pixel's location)
		samplePoint = mul( rotation, samplePoint );
		// Correct for the aspect ratio (since we're doing this in screen space)
		// (Note that this has to happen _after_ the rotation!)
		samplePoint.y *= aspectRatio;
		// Now we want to scale it based on the radius,
		float2 sampleOffset = samplePoint * radius;
		float sampleWidth = sphereWidths[j] * radii;

		// Now we can sample our pair of points
		float2 UvSample1 = UV0.xy + sampleOffset.xy;
		float depthSample1 = READ_CONVERTED_DEPTH(DepthSpecBuffer, DepthSpecBufferSampler, UvSample1 );
		float2 UvSample2 = UV0.xy - sampleOffset.xy;
		float depthSample2 = READ_CONVERTED_DEPTH(DepthSpecBuffer, DepthSpecBufferSampler, UvSample2 );

		// Scale to get them in view space
		float2 depthSamples = float2( depthSample1, depthSample2 ) + float2(0.001f * fDepth, 0.001f * fDepth);
    depthSamples *= g_FarPlaneZ;
		
		// Find the differences between the samples and our reference sample
		float2 depthDifferences = depth.xx - depthSamples;
		// How much of the sphere do the samples occlude?
		//float2 occlusionContributions = saturate( ( depthDifferences / sampleWidth.xx ) + float2(0.5f, 0.5f) );
		float2 occlusionContributions;
		occlusionContributions.x = saturate( ( depthDifferences.x / sampleWidth ) + 0.5f );
		occlusionContributions.y = saturate( ( depthDifferences.y / sampleWidth ) + 0.5f );

		// We want to modify the occlusion by distance,
		// so that objects don't occlude other objects after a certain distance
		float2 distanceModifiers;
		distanceModifiers.x = saturate( ( maxDistance - depthDifferences.x ) / maxDistance );
		distanceModifiers.y = saturate( ( maxDistance - depthDifferences.y ) / maxDistance );
		// If the occluder is too far in front to be useful, try to use the other samples inverted occlusion
		// (Since this is a good approximation of flat surfaces)
		// If that is too far in front to be useful, just default to 0.5
		
		float2 modifiedContributions = lerp(
			lerp(float2(0.5f, 0.5f), float2(1.0f, 1.0f) - occlusionContributions.yx, distanceModifiers.yx),
			occlusionContributions.xy,
			distanceModifiers.xy);
		// Weight the contributions
		
		modifiedContributions *= sphereWeights[j];
		
		// Add them to our total
		occlusionAmount += (modifiedContributions.x + modifiedContributions.y);
	}
	
	return occlusionAmount;
}

float4 ps_main( PS_IN In ) : SV_Target
{
	float g_FarPlaneZ = In.ClipPlanes.y;
	float g_ssaoRadiusInner = fOcclusionParams.x;
	float g_ssaoRadiusOuter = fOcclusionParams.y;
	
	float g_ssaoMaxDistanceInner = fOcclusionParams.z;
	float g_ssaoMaxDistanceOuter = fOcclusionParams.w;
 	
	float centerWeight = 0.04906672;

	float depth = READ_CONVERTED_DEPTH(DepthSpecBuffer, DepthSpecBufferSampler, In.UV0.xy);
	clip(0.999f - depth);
  
	float aspectRatio = In.UV0.z;
	// Get the rotation data
	float2 rotationSinCos = vTex2D(NoiseNormalMap, NoiseNormalMapSampler, In.UV1).rg * 2.0f - 1.0f;
    float2x2 rotation =
	{
		{ rotationSinCos.y, rotationSinCos.x },
		{ -rotationSinCos.x, rotationSinCos.y }
	};

	// Count this as our first sample point:
	// We know that it's half occluded by definition
	float occlusionAmount = 0.5f * centerWeight;

	// Go through each radius to find out how occluded it is
	$if NUM_RADII == 2
		float2 occlusionAmounts = float2( occlusionAmount, occlusionAmount );
		occlusionAmounts[0] += CalculateRadius(g_ssaoRadiusInner, depth, rotation, aspectRatio, In.UV0.xy, g_FarPlaneZ, g_ssaoMaxDistanceInner);
		occlusionAmounts[1] += CalculateRadius(g_ssaoRadiusOuter, depth, rotation, aspectRatio, In.UV0.xy, g_FarPlaneZ, g_ssaoMaxDistanceOuter);
	
		// We don't want anything below 0.5, and we want to normalize to [0,1]
		occlusionAmounts = saturate( ( occlusionAmounts - fThreshold.xx ) * fThreshold.yy );
		
		// Combine
		// I think additive gives a really nice look, but you could also take the max
		occlusionAmount = 1.0f - saturate( occlusionAmounts.x + occlusionAmounts.y );
	$else
		float occlusionAmounts = occlusionAmount;
		occlusionAmounts += CalculateRadius(g_ssaoRadiusOuter, depth, rotation, aspectRatio, In.UV0.xy, g_FarPlaneZ, g_ssaoMaxDistanceOuter);
		
		// We don't want anything below 0.5, and we want to normalize to [0,1]
		occlusionAmounts = saturate( ( occlusionAmounts - fThreshold.x ) * fThreshold.y );
		occlusionAmount = 1.0f - occlusionAmounts;
	$endif

	occlusionAmount = lerp(1.0f, occlusionAmount, fAmount);
	return float4(occlusionAmount.x, occlusionAmount.x, occlusionAmount.x, 1.0f);
}


