#include <Shaders/VisionCommon.inc>
#include <Shaders/ShaderHelpers.inc>
#include "DeferredShadingHelpers.inc"

#ifdef LIGHTMAP
  #define VMATERIAL_USELIGHTMAPS
#endif

#ifdef LIGHTMAPDOT3
  #define VMATERIAL_DOT3LIGHTMAPS
#endif  
  
#include <Shaders/TerrainShadersCommon.inc>


#ifdef _VISION_DX11
cbuffer g_GlobalConstantBufferFrame : register (b0)
{
  float4   contextCP       : packoffset(c12);  // context clip planes: near, far, 1/(far-near)
  float4   eyePos      : packoffset(c17);  // eye position
  float4   depthFog        : packoffset(c19);  // depth fog param
  float4   depthFogCol     : packoffset(c20);  // depth fog color
  float4   gAmbientCol     : packoffset(c21);  // global ambient color
  float4   _NoExport_lightmapMul : packoffset(c22);  // lightmap multiplier (skalar)
}
cbuffer g_GlobalConstantBufferObject : register (b1)
{
  float4x4 matMV       : packoffset(c0);   // model view matrix
  float4x4 matMVP      : packoffset(c4);   // model view projection matrix
  float4   MUVtoLM     : packoffset(c13);  // model UV to lightmap
}
cbuffer g_GlobalConstantBufferUserPS : register (b2)
{
  float4 MaterialParams : packoffset(c0);
  float3 MaterialAmbient : packoffset(c1);
}

#else
  float4x4 matMV : register(c0);
  float4x4 matMVP : register(c8);
  float4   contextCP : register(c24);
  float3   eyePos : register(c31);

  float4   _NoExport_lightmapMul: register(c0);

  // vertex shader constants  
  float4 DetailPassTiling  : register(c42); // for a single detail texture pass this is the tiling factor
  float4 MUVtoLM           : register(c60); // transforms global lightmap coords to page
  float4 depthFog          : register(c62);

  // pixel shader constants
  float4   depthFogCol      : register(c1);

  // Material parameters and ambient
  float3 gAmbientCol : register(c21);
  float4 MaterialParams : register(c32);
  float3 MaterialAmbient : register(c33); 

#endif

struct VS_OUT
{
#ifdef MSAA
  centroid float4 ProjPos  : SV_Position;
#else
  float4 ProjPos  : SV_Position;
#endif

  float4 baseUV : TEXCOORD0;  // xy=global UV, zw=sector UV
 	
#if defined(LIGHTMAP) || defined(LIGHTMAPDOT3)
  float2 LightmapUV : TEXCOORD1;
#endif

  float4 EyeDirAndDepth : TEXCOORD2;
  
  float4 WorldUV : TEXCOORD3;
  
  #if defined(SINGLE_DETAIL_PASS)
	float4 SinglePassDetailPassTiling : TEXCOORD4;
  #endif
};

float3 GetVectorInWorldSpace(float3 InVector, float3 Tangent, float3 Normal, float3 BiNormal)
{
  float3 OutVector;
   
  OutVector.x = dot (float3(Tangent.x, -BiNormal.x, Normal.x), InVector);
  OutVector.y = dot (float3(Tangent.y, -BiNormal.y, Normal.y), InVector);
  OutVector.z = dot (float3(Tangent.z, -BiNormal.z, Normal.z), InVector);
 
  return OutVector;
}

#ifdef _IS_PIXEL_SHADER

#ifdef _VISION_XENON
[maxtempreg(36)]
#endif
PS_OUT ps_main( VS_OUT In )
{
  PixelDataUnpacked pixelData;
  
  float2 normalUV = In.baseUV.zw * _NoExport_NormalmapTransform.xy + _NoExport_NormalmapTransform.zw;
  float3 worldNormal = normalize(vTex2D(Normalmap, NormalmapSampler, normalUV).xyz*2.0f-1.0f);
  float3 worldTangent = float3(1.0f, 0.0f, 0.0f);

  #ifdef USE_3WAY
    TerrainLayerConfig config = CreateTerrainLayerConfig3Way(In.baseUV.zw, In.WorldUV.xyz, worldNormal);
  #else
    TerrainLayerConfig config = CreateTerrainLayerConfig(In.baseUV.zw, In.WorldUV.xyz, worldNormal);
  #endif
    
  float4 diffuse = GetTerrainDiffuseColor(config);
  
  #ifdef FULLBRIGHT
    pixelData.vDiffColor = float3(0.0f, 0.0f, 0.0f);
  #else
    pixelData.vDiffColor = diffuse.xyz;
  #endif

  // ================ NORMAL ==================
#ifdef VMATERIAL_NORMALMAP
  float3 worldBiNormal = cross(worldTangent, worldNormal);
  worldTangent = cross(worldNormal, worldBiNormal); // re-orthogonalize
  float4 mixedNormals = GetTerrainDetailNormal(config);
  float4 normalmap = DecodeNormalAndAlpha(mixedNormals);
  pixelData.vNormal = normalize(GetVectorInWorldSpace(normalmap.xyz, worldTangent, worldNormal, worldBiNormal));
  
  // TODO FOR PARALLAX: Transform Out.EyeDirAndDepth.xyz to worldspace, use for lookup, re-lookup normal + basetex
#else
  pixelData.vNormal = worldNormal;
#endif

  // ================ SPECULAR ==================
#ifdef VMATERIAL_SPECULAR
  pixelData.fSpecExp = mapToNormalizedRangeFrom255(MaterialParams.y);
  pixelData.vSpecColor = saturate(mixedNormals.aaa*MaterialParams.xxx);
#else
  pixelData.fSpecExp = 1.0f;
  pixelData.vSpecColor = float3(0.0f, 0.0f, 0.0f);
#endif

  // ================ AMBIENT ==================
  pixelData.vAccInitial = gAmbientCol.rgb;

  // ================ DEPTH ==================
  pixelData.fDepth = In.EyeDirAndDepth.w;

  // ================ LIGHTMAP ==================
#ifdef LIGHTMAPDOT3
  float4 lightmap1 = vTex2D(Lightmap1, Lightmap1Sampler,In.LightmapUV.xy); 
  float4 lightmap2 = vTex2D(Lightmap2, Lightmap2Sampler,In.LightmapUV.xy); 
  float4 lightmap3 = vTex2D(Lightmap3, Lightmap3Sampler,In.LightmapUV.xy);
  pixelData.vAccInitial += GetLightmapDot3LightingDiffuse(normalmap.xyz, lightmap1, lightmap2, lightmap3) * _NoExport_lightmapMul.x;
#endif
  
#ifdef LIGHTMAP
  pixelData.vAccInitial += vTex2D(Lightmap, LightmapSampler, In.LightmapUV.xy) * _NoExport_lightmapMul.x; 
#endif

  pixelData.vAccInitial *= diffuse.rgb;
  pixelData.fMaterial = 1.0f;
  
  PS_OUT result = packPixelValues(pixelData);  
  
  #ifdef MSAA
		// When one or more samples aren't located in the primitive's interior, the centroid of the remaining samples won't coincide with
		// the fragment's center (which, with the SV_Position semantic, has a fractional part of 0.5 for the xy coordinates).  
		//
		// We exploit this fact to differentiate between fully interior pixels, which will only need to be sampled once during the combine
		// pass, and pixels that need to be fully multi-sampled.  The idea is to compute the distance d (Manhattan distance here, to reduce
		// roundoff errors) between the sample centroid and the fragment center (equivalently, between the fractional part of the sample
		// centroid and the (0.5,0.5) vector).  
		//
		// Resolving the corresponding render target and writing (d > 0 ? 0 : 1) into the stencil buffer then yields the desired mask.
		float edgeResponse = dot(abs(frac(In.ProjPos.xy) - 0.5), 1000000);
		result.Accumulation.a = min (edgeResponse, 0.9);
  #endif  

  return result;
}

#endif // _IS_PIXEL_SHADER


#ifdef _IS_VERTEX_SHADER

VS_OUT vs_main( VS_IN In )
{
   VS_OUT Out;
   float4 wsPos = GetVertexPosition(In);
   Out.ProjPos = mul( matMVP, wsPos );

   Out.WorldUV = GetWorldUV(In);
   Out.baseUV = GetUV(In.ObjPosXY);

#ifdef VMATERIAL_PARALLAX
  Out.EyeDirAndDepth.xyz = eyePos-wsPos.xyz; //GetVectorInTextureSpace(eyePos-wsPos.xyz, Out.Tangent, Out.Normal, Out.BiTangent);
  Out.EyeDirAndDepth.w = mul(matMV, float4(wsPos.xyz, 1.0f)).z / contextCP.y;
#else
  Out.EyeDirAndDepth.xyz = float3(0.0f, 0.0f, 0.0f);
  Out.EyeDirAndDepth.w = mul(matMV, float4(wsPos.xyz, 1.0f)).z / contextCP.y;
#endif

#if defined(LIGHTMAP) || defined(LIGHTMAPDOT3)
	Out.LightmapUV = Out.baseUV.zw * MUVtoLM.xy + MUVtoLM.zw; // per sector UV to lightmap
#endif

#if defined(_VISION_PS3) || defined(_VISION_WIIU)
  Out.EyeDirAndDepth.w = -Out.EyeDirAndDepth.w;
#endif

   return Out;
}

#endif //_IS_VERTEX_SHADER
