//
#include <Shaders/VisionCommon.inc>
#include <Shaders/DeferredShadingHelpers.inc>

#include <Shaders/FogHelpers.inc>

#ifdef STANDALONE

struct PS_IN
{
  float4   ProjPos     : SV_Position;
  float4   CameraPos   : TEXCOORD0;
  float4   ClipPlanes  : TEXCOORD1;
  float4   FogCoord    : TEXCOORD2;
  
  #if defined(_VISION_PS3) || defined(_VISION_PSP2)
    float2 UV0 : WPOS;
  #elif defined(_VISION_DX11)
    float4 UV0 : VPOS_MANUAL;
  #elif defined(_VISION_XENON)
    float4 UV0 : TEXCOORD3;
  #else
    float2 UV0 : VPOS;
  #endif
};

#ifdef _VISION_DX11
cbuffer g_GlobalConstantBufferFrame : register (b0)
{
  float4 DepthColor            : packoffset(c20);
  float4 LayeredFogParams      : packoffset(c26);  // x = HeightFalloff, y = GroundOFfset, zw = FogExtents
  float4 LayeredColorDensity   : packoffset(c27);  // xyz = BaseColor, w = Density
}

cbuffer g_GlobalConstantBufferUser : register (b2)
{
  float3 UpperLeftCorner       : packoffset(c17);
  float3 RightDir              : packoffset(c18);
  float3 DownDir               : packoffset(c19);  
  float2 InvTexSizeMask        : packoffset(c20);
}

#elif defined(_VISION_PS3) || defined(_VISION_PSP2) || defined(_VISION_GLES2) || defined(_VISION_WIIU)
  float4 DepthColor            : register(c1);
  
  float4 LayeredFogParams      : register(c16);
  float4 LayeredColorDensity   : register(c17);
  
  float3 UpperLeftCorner       : register(c50);
  float3 RightDir              : register(c51);
  float3 DownDir               : register(c52);
  float2 InvTexSizeMask        : register(c53);

#else
  float4 DepthColor            : register(c1);

  float4 LayeredFogParams      : register(c16);
  float4 LayeredColorDensity   : register(c17);
  
  float3 UpperLeftCorner;
  float3 RightDir;
  float3 DownDir;
  float2 InvTexSizeMask;
#endif

$if MSAA_SAMPLES > 1
  V_REQUIRES_SM40
  #define InputBuffer Texture2DMS<float, $MSAA_SAMPLES>
$else
  #define InputBuffer Texture2D<float>
$endif

#ifdef _VISION_DX11
  InputBuffer DepthTexture                  : register(t0);
  sampler     DepthTextureSampler           : register(s0);
#else
  sampler2D   DepthTexture                  : register(s0);
#endif

#ifdef _VISION_DX11
#else
  
#endif


float2 GetScreenPos(PS_IN In)
{
  float2 screenPos;
  
  #ifdef _VISION_DX11
    screenPos = (In.UV0.xy / In.UV0.w) * 0.5f + 0.5f;
  #elif defined(_VISION_XENON)
    screenPos = (In.UV0.xy * 0.5f + 0.5f) + (0.5f * InvTexSizeMask);
  #elif defined(_VISION_PS3)
    screenPos.x = In.UV0.x * InvTexSizeMask.x;
    screenPos.y = 1.0f - (In.UV0.y * InvTexSizeMask.y);
  #else
    screenPos = In.UV0.xy * InvTexSizeMask + 0.5f * InvTexSizeMask;
  #endif
 
  return screenPos;
}

float4 ps_main( PS_IN In ) : SV_Target
{
  float2 screenPos = GetScreenPos(In);

  // Compute the world position and view vector
  float Opacity = 0;
  
  float4 A = float4(0,0,0,0);
  float4 B = float4(0,0,0,0);
  
  float scrDepth;
  
$ifdef LINEAR

  A.xyz = DepthColor.xyz;
  
  $if MSAA_SAMPLES > 1
    for (int i = 0; i < $MSAA_SAMPLES; i++)
    {
      scrDepth = DepthTexture.Load (int2(In.ProjPos.xy), i);
      A.w += ComputeDepthFogOpacity(scrDepth, In.FogCoord, In.ClipPlanes);
    }
  
    A.w /= $MSAA_SAMPLES;
    
  $else

    scrDepth = READ_CONVERTED_DEPTH(DepthTexture, DepthTextureSampler, screenPos);
    A.w = ComputeDepthFogOpacity(scrDepth, In.FogCoord, In.ClipPlanes);
    
  $endif
  
  A.w = saturate(A.w);
  
$endif

$if EXPONENTIAL

  $if MSAA_SAMPLES > 1
    // Simply use the first sample, since otherwise we'd be doing three exp instructions per sample
    scrDepth = DepthTexture.Load(int2(In.ProjPos.xy), 0);
  $else
    scrDepth = READ_CONVERTED_DEPTH(DepthTexture, DepthTextureSampler, screenPos);
  $endif
  
  B.xyz = LayeredColorDensity.xyz;
  
  float3 ViewDirection = scrDepth * (UpperLeftCorner + RightDir * screenPos.x + DownDir * screenPos.y);
  B.w = saturate(ComputeLayeredFogOpacity(ViewDirection, LayeredFogParams, LayeredColorDensity.w, In.ClipPlanes, In.CameraPos.xyz));
  
$endif

  // We want the final color c' to be the result of applying the depth fog first, followed by layered fog, 
  // using alpha blending both times.  Writing this out,
  // 
  //   c' = a1 * f1 + (1 - a1) * (a2 * f2 + (1 - a2) * c),
  // 
  // where c is the old color, a1/a2 are the opacities, and f1/f2 are the colors.  
  // Expanding,
  //
  //   c' = a1 * f1 + (1 - a1) * a2 * f2 + (1 - a1) * (1 - a2) * c.
  //
  // To modulate the backbuffer color c, we set the target blend to Src_Alpha and write (1-a1)(1-a2) to the alpha channel.  
  // To modulate the output color, we simply set the source blend to One and premodulate in the shader.

  return MixFog(A, B);
}

#endif
