#include <Shaders/VisionCommon.inc>
#include <Shaders/ShaderHelpers.inc>
#include "DeferredShadingHelpers.inc"

#ifdef _VISION_DX11
  Texture2D <float4> BaseTexture             : register(t0);
  sampler            BaseTextureSampler      : register(s0);
  $ifdef VMATERIAL_NORMALMAP
    Texture2D <float4> NormalMap          : register(t1);
    sampler            NormalMapSampler   : register(s1);
  $endif
  $ifdef VMATERIAL_REFMAP
    Texture2D <float4> RefMap             : register(t2);
    sampler            RefMapSampler      : register(s2);
  $endif
  Texture2D <float4> DecalTexture            : register(t3);
  sampler            DecalTextureSampler     : register(s3);
  Texture2D <float4> DecalMaskTexture        : register(t4);
  sampler            DecalMaskTextureSampler : register(s4);
  $ifdef LIGHTMAP
    Texture2D <float4> LightMap           : register(t5);
    sampler            LightMapSampler    : register(s5);
  $endif
  $ifdef LIGHTMAPDOT3
    Texture2D <float4> LightMap1          : register(t5);
    sampler            LightMapSampler1   : register(s5);
    Texture2D <float4> LightMap2          : register(t6);
    sampler            LightMapSampler2   : register(s6);
    Texture2D <float4> LightMap3          : register(t7);
    sampler            LightMapSampler3   : register(s7);
  $endif
  $ifdef VMATERIAL_GLOWMAP
    Texture2D <float4> GlowMap            : register(t8);
    sampler            GlowMapSampler     : register(s8);
  $endif
  
  cbuffer g_GlobalConstantBufferFrame     : register (b0)
  {
    float4    gAmbientCol                 : packoffset(c21);  // global ambient color
    float4    lightmapMul                 : packoffset(c22);  // lightmap multiplier (scalar) 
  }
  
  cbuffer g_GlobalConstantBufferUser      : register (b2)
  {
    float4 MaterialParams                 : packoffset(c0);
    float3 MaterialAmbient                : packoffset(c1);
 
    float  fAlphaTestThreshold            : packoffset(c2);
    float  GlobalAmbientMultiplier        : packoffset(c3);
    
    float  gMaterialIndex : packoffset(c4);
  
  $ifdef VMATERIAL_GLOWMAP  
    float3 GlowColor                      : packoffset(c5);
    float GlowIntensity                   : packoffset(c6);  
  $endif

    float  fMaskTestThreshold             : packoffset(c7);
  }  
    
  $ifdef LIGHTGRID
    cbuffer g_GlobalConstantBufferLightGrid : register (b3)
    {
      float4      Light0                  : packoffset(c0);
      float4      Light1                  : packoffset(c1);
      float4      Light2                  : packoffset(c2);
      float4      Light3                  : packoffset(c3);
      float4      Light4                  : packoffset(c4);
      float4      Light5                  : packoffset(c5);
    }
  $endif
  
#else
  sampler2D BaseTexture                   : register(s0);

  $ifdef VMATERIAL_NORMALMAP
    sampler2D NormalMap                   : register(s1);
  $endif

  $ifdef VMATERIAL_REFMAP
    sampler2D RefMap                      : register(s2);
  $endif

  sampler2D DecalTexture                  : register(s3);
  sampler2D DecalMaskTexture              : register(s4);

  $ifdef LIGHTMAP
    sampler2D LightMap                    : register(s5);
  $endif

  $ifdef LIGHTMAPDOT3
    sampler2D LightMap1                   : register(s5);
    sampler2D LightMap2                   : register(s6);
    sampler2D LightMap3                   : register(s7);
  $endif
  


  float4 lightmapMul : register(c0);

  float3 gAmbientCol : register(c21);
  float4 MaterialParams : register(c32);
  float3 MaterialAmbient : register(c33);

  float  fAlphaTestThreshold : register(c34);
  float  GlobalAmbientMultiplier : register(c35);
  
  float  gMaterialIndex : register(c36);
  
  $ifdef LIGHTGRID 
    float4 Light0 : register(c26);
    float4 Light1 : register(c27);
    float4 Light2 : register(c28);
    float4 Light3 : register(c29);
    float4 Light4 : register(c30);
    float4 Light5 : register(c31);
  $endif
  
  $ifdef VMATERIAL_GLOWMAP
    sampler2D GlowMap           : register(s6);
    float3 GlowColor            : register(c37);
    float GlowIntensity         : register(c38);
  $endif

    float  fMaskTestThreshold   : register(c39);
#endif



struct PS_IN                              
{                                             
  $ifdef MSAA
		V_REQUIRES_SM41
    centroid float4 ProjPos : SV_Position;
  $else
    float4 ProjPos : SV_Position;
  $endif
  
  float4 UV01 : TEXCOORD0;
  float3 Normal  : TEXCOORD1;

  $ifdef VMATERIAL_NORMALMAP
    float3 Tangent : TEXCOORD2;
    float3 BiTangent : TEXCOORD3;
  $endif

  float4 EyeDirAndDepth : TEXCOORD4;

  $ifdef VERTEXCOLOR
    float4 VertColor : TEXCOORD5;
  $endif
};                         


PS_OUT ps_main( PS_IN In )
{
  PixelDataUnpacked pixelData;

  $if defined (VMATERIAL_PARALLAX) || defined (LIGHTMAPDOT3) || defined (LIGHTGRID)
    float3 eyeNorm = normalize(In.EyeDirAndDepth.xyz);
  $endif  
  
  $ifdef VMATERIAL_PARALLAX
    float fHeight = vTex2D(NormalMap, NormalMapSampler, In.UV01.xy).a * MaterialParams.z - MaterialParams.w;
    float2 BaseMapCoord = In.UV01.xy - (eyeNorm.xy * fHeight);
    $ifdef VMATERIAL_PARALLAX_REFINEMENT
      fHeight += vTex2D(NormalMap, NormalMapSampler, BaseMapCoord).a * MaterialParams.z - MaterialParams.w;
      BaseMapCoord = In.UV01.xy - (eyeNorm.xy * fHeight * 0.5f);    
    $endif
  $else
    float2 BaseMapCoord = In.UV01.xy;
  $endif
    
  float2 MaskMapCoord = In.UV01.zw;
  float4 cBaseTex  = vTex2D(BaseTexture     , BaseTextureSampler, BaseMapCoord);
  float4 cDecalTex = vTex2D(DecalTexture    , BaseTextureSampler, BaseMapCoord);
  float4 cMaskTex  = vTex2D(DecalMaskTexture, BaseTextureSampler, MaskMapCoord);
  
  $ifdef ALPHATEST
    clip(cBaseTex.a - fAlphaTestThreshold);
    clip(fMaskTestThreshold - cMaskTex.r);
  $endif

  cBaseTex.xyz = lerp( cDecalTex.xyz, cBaseTex.xyz, cMaskTex.r );

  $ifdef FULLBRIGHT
    pixelData.vDiffColor = float3(0.0f, 0.0f, 0.0f);
  $else
    pixelData.vDiffColor = cBaseTex.xyz;
  $endif

  $ifdef VMATERIAL_NORMALMAP
    float4 cNormalMap = DecodeNormalAndAlpha(vTex2D(NormalMap, NormalMapSampler, BaseMapCoord));
    cNormalMap.xyz = normalize(cNormalMap.xyz);
    float3 worldNormal = GetWorldNormal(cNormalMap, In.Tangent, In.BiTangent, In.Normal);
  $else
    float3 worldNormal = normalize(In.Normal);
    float4 cNormalMap = float4(worldNormal, 1.0f);
  $endif

  $ifdef DEFERRED_OUTPUT
    pixelData.vNormal = worldNormal;
  $endif

  $ifdef VMATERIAL_REFMAP
    float4 SpecMul = vTex2D(RefMap, RefMapSampler, BaseMapCoord) * MaterialParams.xxxy;    //specmul masked by specmap
    float fSpecExp = SpecMul.a;
    pixelData.fSpecExp = mapToNormalizedRangeFrom255(fSpecExp);
    pixelData.vSpecColor = saturate(SpecMul.rgb);
  $else
    float4 SpecMul = MaterialParams.xxxy;
    float fSpecExp = MaterialParams.y;
    
    $ifdef FULLBRIGHT
      pixelData.fSpecExp = 1.0f;
      pixelData.vSpecColor = float3(0.0f, 0.0f, 0.0f);
    $else
      $ifdef VMATERIAL_NORMALMAP
        SpecMul *= cNormalMap.w; // specmul masked by normalmap.a
      $endif
      
      pixelData.fSpecExp = mapToNormalizedRangeFrom255(fSpecExp);
      pixelData.vSpecColor = saturate(SpecMul.xyz);
    $endif
  $endif
  
  $ifdef LIGHTMAP
    // ordinary lightmap lighting
    pixelData.vAccInitial = (vTex2D(LightMap, LightMapSampler, In.UV01.zw).xyz*lightmapMul.x + gAmbientCol.xyz*GlobalAmbientMultiplier) * cBaseTex.xyz;
  $elif defined (LIGHTMAPDOT3)
    // lightmap dot3 lighting
    float4 cLightmap1 = vTex2D(LightMap1, LightMapSampler1, In.UV01.zw);
    float4 cLightmap2 = vTex2D(LightMap2, LightMapSampler2, In.UV01.zw);
    float4 cLightmap3 = vTex2D(LightMap3, LightMapSampler3, In.UV01.zw);
    const float3 vLightDir1 = float3(-0.816497, -0.000000, 0.577350);
    const float3 vLightDir2 = float3(0.408248,  -0.707107, 0.577350);
    const float3 vLightDir3 = float3(0.408248,   0.707107, 0.577350);
    LIGHT_RESULT_SPEC difspec, lsres;    
    difspec = ComputeDot3LightmapLightingSpec(cNormalMap/*worldNormal*/, vLightDir1, eyeNorm, cLightmap1, fSpecExp);
    lsres = ComputeDot3LightmapLightingSpec(cNormalMap/*worldNormal*/, vLightDir2, eyeNorm, cLightmap2, fSpecExp);
    difspec.diffuse  += lsres.diffuse;
    difspec.specular += lsres.specular;
    lsres = ComputeDot3LightmapLightingSpec(cNormalMap/*worldNormal*/, vLightDir3, eyeNorm, cLightmap3, fSpecExp);
    difspec.diffuse  += lsres.diffuse;
    difspec.specular += lsres.specular;
    difspec.diffuse.xyz *= lightmapMul.x;
    difspec.specular.xyz *= lightmapMul.x;
    pixelData.vAccInitial = (difspec.diffuse.xyz+gAmbientCol.xyz*GlobalAmbientMultiplier) * cBaseTex.xyz + difspec.specular.xyz*SpecMul;
  $elif defined (LIGHTGRID)
    LIGHT_RESULT_SPEC res = GetLightGridSpec_WS(eyeNorm, worldNormal, Light0, Light1, Light2, Light3, Light4, Light5, fSpecExp);
    pixelData.vAccInitial = (res.diffuse.xyz + gAmbientCol.xyz*GlobalAmbientMultiplier)*cBaseTex.xyz + res.specular*SpecMul;
    //pixelData.vAccInitial = pixelData.vAccInitial * 0.001f + cBaseTex.xyz;
  $elif defined (VERTEXCOLOR) 
    // no lightmap with vertex color
    pixelData.vAccInitial = (In.VertColor + gAmbientCol*GlobalAmbientMultiplier) * cBaseTex.xyz;
  $else
    pixelData.vAccInitial = (MaterialAmbient + gAmbientCol*GlobalAmbientMultiplier) * cBaseTex.xyz;
  $endif

  pixelData.fDepth = In.EyeDirAndDepth.w;

  $if defined (VMATERIAL_PARALLAX) && defined (VMATERIAL_PARALLAX_DEPTH_DISPLACEMENT)
    pixelData.fDepth += fHeight * eyeNorm.z * 0.1f;
  $endif
  
  pixelData.fMaterial = gMaterialIndex;
  
  $ifdef VMATERIAL_GLOWMAP
    float3 cGlowTex = vTex2D(GlowMap, BaseTextureSampler, BaseMapCoord).rgb;
    pixelData.vAccInitial.rgb += cGlowTex * GlowColor.rgb * GlowIntensity;
  $endif
  
  PS_OUT result = packPixelValues(pixelData);  
  
  $ifdef MSAA
    $ifdef ALPHATEST
      float edgeResponse =  max (0.9, 1.0 - min (1.0, dot(abs(frac(In.ProjPos.xy) - 0.5), 1000)));
        
      result.Accumulation.a = pow (max (0.0, (cBaseTex.a - fAlphaTestThreshold) / (1.0 - fAlphaTestThreshold)), 2.0) * edgeResponse;
    $else
      // When one or more samples aren't located in the primitive's interior, the centroid of the remaining samples won't coincide with
      // the fragment's center (which, with the SV_Position semantic, has a fractional part of 0.5 for the xy coordinates).  
      //
      // We exploit this fact to differentiate between fully interior pixels, which will only need to be sampled once during the combine
      // pass, and pixels that need to be fully multi-sampled.  The idea is to compute the distance d (Manhattan distance here, to reduce
      // roundoff errors) between the sample centroid and the fragment center (equivalently, between the fractional part of the sample
      // centroid and the (0.5,0.5) vector).  
      //
      // Resolving the corresponding render target and writing (d > 0 ? 0 : 1) into the stencil buffer then yields the desired mask.
      float edgeResponse = dot(abs(frac(In.ProjPos.xy) - 0.5), 1000000);
        
      result.Accumulation.a = min (edgeResponse, 0.9);
    $endif
  $endif
  
  return result;
}



