//
#include <Shaders/VisionCommon.inc>

#define EPSILON 0.00001f

struct PS_IN
{
  float4 ProjPos  : SV_Position;
  float2 UV       : TEXCOORD0;
};

// Indices for coefficient vectors (e.g., Scattering[RAYLEIGH])
#define RAYLEIGH 0
#define MIE 1

#ifdef _VISION_DX11
  Texture2D <float4> OpticalDepthTexture : register(t0);
  sampler            OpticalDepthSampler : register(s0);
#else
  sampler2D          OpticalDepthTexture : register(s0);
#endif


#ifdef _VISION_DX11
cbuffer g_GlobalConstantBufferObject : register (b1)
{
  float4x3 MatModelView : packoffset(c0);
}
cbuffer g_GlobalConstantBufferUser : register (b2)
{
  float3 CameraPosition       : packoffset(c0);  // camera position in world space, height relative to the center of the earth
  float4 ScatteringR          : packoffset(c1);  // Rayleigh scattering coefficients (beta_r).  The w component is a scaling factor.
  float4 ScatteringM          : packoffset(c2);  // Mie scattering coefficients (beta_m).  The w component is a scaling factor.
  float2 MeanDensityHeight    : packoffset(c3);  // the elevation at which the molecular or aerosol density equals the atmospheric mean
  float3 SunDirection         : packoffset(c4);  // sun direction
  float2 AtmosphereBoundaries : packoffset(c5);  // boundaries of the atmosphere layer (Earth radius below)
  float  MieFactor            : packoffset(c6);  // mie factor
}

#elif defined(_VISION_PS3) || defined(_VISION_PSP2) || defined(_VISION_GLES2) || defined(_VISION_WIIU)

  float3 CameraPosition       : register(c32);
  float4 ScatteringR          : register(c33);
  float4 ScatteringM          : register(c34);
  float2 MeanDensityHeight    : register(c35);
  float3 SunDirection         : register(c36);
  float2 AtmosphereBoundaries : register(c37);
  float  MieFactor            : register(c38);

#else

  float3 CameraPosition;
  float4 ScatteringR;
  float4 ScatteringM;
  float2 MeanDensityHeight;
  float3 SunDirection;
  float2 AtmosphereBoundaries;
  float  MieFactor;
#endif

#define PI 3.1415926535897932384626433832795f

float3 SphericalToCartesian(float2 coords)
{
  float2 cosT;
  float2 sinT;
  sincos(coords, sinT, cosT);
  return float3(cosT.x*sinT.y,sinT.x*sinT.y,cosT.y);
}


// Computes the intersection of a ray with either the earth (radius R[0]) or the atmosphere (radius R[1]),
// where the ray is given by the height h and the cosine of the zenith angle cosT.
float RayEndpoint(float h, float cosT, float2 R) 
{
  // The below follows from the quadratic |p+xv|^2-r^2=0 for x, where p = (0,h) and v = (sin(t),cos(t)).
  float e      = (cosT * cosT - 1) * h * h;
  float b      = h * cosT;
  float2 discr = e + R * R;

  // Find the intersection point with the atmosphere
  float d = -b + sqrt(max(0,discr[1]));
  
  // Does the ray also intersect the earth?
  if (discr[0] > 0.0) 
  {
    float dE = -b - sqrt(discr[0]);
    
    // At a positive distance?
    if (dE > -0.01f && d > dE) 
    {
      // Yes, clip it.
      d = dE;
    }
  }
  return d;
}

#define NUM_SAMPLES 16

float4 GetOpticalDepth(float r, float cosT)
{
  float u = (r - AtmosphereBoundaries[0]) / (AtmosphereBoundaries[1] - AtmosphereBoundaries[0]);
  float v = 0.5f - cosT * 0.5f;
  return vTex2D(OpticalDepthTexture, OpticalDepthSampler, float2(u, v));
}

float RayleighPhase(float cosTheta)
{
  return 0.75f * (1 + cosTheta * cosTheta);
}

float MiePhase(float cosTheta, float g)
{
  return (3 * (1 - g*g)) / (2 * (2 + g*g)) * (1 + cosTheta * cosTheta) / pow((1 + g*g - 2 * g * cosTheta), 1.5f);
}

struct PS_OUT
{
  float4 Color          : SV_Target0;
  float4 Color_Overcast : SV_Target1;
};

#define IntegrationStep(_v_, _p_, _height_, _dp_, _R_, _M1_, _M2_, _tP_start_, _rcp_, _dx_, _result_) \
{ \
  float4 tS = GetOpticalDepth(_height_, dot(_p_, SunDirection) / _height_); \
  float4 tP = GetOpticalDepth(_height_, dot(_p_, _v_) / _height_); \
  tP.xy = _tP_start_.xy - tP.xy; \
  \
  _p_ += _dp_; \
  _height_ = length(_p_); \
  \
  /* Transmittance:  T = exp(-t(View)-t(Sun)), where t is the wavelength-dependent optical depth */ \
  float3 t1 = ScatteringR.w * tP[0] * ScatteringR.xyz + ScatteringM.w * tP[1] * ScatteringM.xyz; \
  float3 t2 = ScatteringR.w * tS[0] * ScatteringR.xyz + ScatteringM.w * tS[1] * ScatteringM.xyz; \
  float3 T  = exp(-t1-t2); \
  \
  /* tS.zw = tP.zw = exp(-altitude / MeanDensityHeights), but tS includes the shadow term from the sun */ \
  float3 scale1 = _R_ * tS[2] * _rcp_[0] + _M1_ * tS[3] * _rcp_[1]; \
  float3 scale2 = _R_ * tS[2] * _rcp_[0] + _M2_ * tS[3] * _rcp_[1]; \
  \
  _result_.Color.xyz += T * scale1 * _dx_; \
  _result_.Color_Overcast.xyz += T * scale2 * _dx_; \
}

PS_OUT ps_main( PS_IN In )
{
  // view direction
  float3 v = SphericalToCartesian(In.UV.xy * PI * float2(2,1));
  
  // view position
  float3 p = CameraPosition;
  
  // cosine of the scattering angle
  float cosT = dot(v, SunDirection);
  
  // starting height
  float height = length(p);
  float startHeight = height;
  
  // distance along the view vector
  float viewDist = RayEndpoint(height, dot(p, v) / height, AtmosphereBoundaries.xy);
  
  // position step
  float3 dp = v * viewDist / NUM_SAMPLES;
  
  float phaseR = RayleighPhase(cosT);
  float phaseM = MiePhase(cosT, MieFactor);

  float3 R   = ScatteringR.w * ScatteringR.xyz * phaseR;
  float3 M1  = ScatteringM.w * ScatteringM.xyz * phaseM;
  float3 M2  = ScatteringM.w * ScatteringM.xyz * phaseR;

  float4 tP_start = GetOpticalDepth(height, dot(p, v) / height);
  
  float2 rcp = (tP_start.zw > 0.1f) ? 1 / tP_start.zw : 1;

  // integrate
  PS_OUT result;
  result.Color = 0;
  result.Color_Overcast = 0;
  
  // first sample, weighted by 0.5f
  IntegrationStep(v, p, height, dp, R, M1, M2, tP_start, rcp, 0.5f, result);
  
  // other samples, weighted by 1.0f
  for (int k = 1; k < NUM_SAMPLES; k++)
  {
    IntegrationStep(v, p, height, dp, R, M1, M2, tP_start, rcp, 1.0f, result);
  }
  
  // last sample, weighted by 0.5f
  IntegrationStep(v, p, height, dp, R, M1, M2, tP_start, rcp, 0.5f, result);
  
  result.Color *= viewDist / NUM_SAMPLES;
  result.Color_Overcast *= viewDist / NUM_SAMPLES;
  return result;
}
