//
#include <Shaders/VisionCommon.inc>
#include <Shaders/ShaderHelpers.inc>

#ifdef _VISION_DX11
cbuffer g_GlobalConstantBufferFrame : register (b0)
{
  float4 depthFog : packoffset(c19);  // linear depth fog parameter: near, far, 1/(far-near)
}

cbuffer g_GlobalConstantBufferObject : register (b1)
{
  float4x4 matMV   : packoffset(c0);   // model view matrix
  float4x4 matMVP  : packoffset(c4);   // model view projection matrix
  float4x4 matMtoW : packoffset(c8);   // model to worldspace matrix
  float4 texTRF : packoffset(c12);  
#ifdef TERRAIN
  float4 MUVtoLM : packoffset(c13);
#else
  float4 modelUV2LM : packoffset(c13);
#endif
}
#else
  float4x4 matMV   : register(c0);
  float4x4 matMVP  : register(c8);
  float4x4 matMtoW : register(c20);
  float4 texTRF : register(c28);
#ifdef TERRAIN
  float4 MUVtoLM : register(c60);
#else
  float4 modelUV2LM : register(c60);
#endif

  float4 depthFog : register(c62);  // linear depth fog parameter: near, far, 1/(far-near)
#endif

#ifdef TERRAIN
  #ifdef _VISION_DX11
  cbuffer g_GlobalConstantBufferTerrain : register (b3)
  {
    float4   SectorScaleOfs : packoffset(c0);   // maps incoming sector xy [0..1] to world space xy position; xy: scaling, zw:ofs
    float4   UVScaleOfs                 : packoffset(c1); // maps incoming [0..1] sector UV range to global [0..1] range; xy: scaling, zw:ofs
    float4   UVSectorScaleOfs           : packoffset(c2); // maps incoming [0..1] mesh UV range to sector [0..1] range; xy: scaling, zw:ofs
  }
  #else
    float4   SectorScaleOfs : register(c38); // maps incoming sector xy [0..1] to world space xy position; xy: scaling, zw:ofs
    float4   UVScaleOfs       : register(c39); // maps incoming [0..1] sector UV range to global [0..1] range; xy: scaling, zw:ofs
    float4   UVSectorScaleOfs : register(c40); // maps incoming [0..1] mesh UV range to sector [0..1] range; xy: scaling, zw:ofs
  #endif
#endif

#ifndef USE_SHADOWMAP
#if defined(_VISION_PS3) || defined(_VISION_PSP2) || defined(_VISION_GLES2) || defined(_VISION_WIIU)
  float4 LightDirPos : register(c64);
#elif defined(_VISION_DX11)
cbuffer g_GlobalConstantBufferUser : register (b2)
{
  float4 LightDirPos : packoffset(c0);
}
#else
  float4 LightDirPos;
#endif
#endif

#ifdef TERRAIN
struct VS_IN
{ 
  float2 ObjPosXY : POSITION; // shared stream
  float ObjPosZ : TEXCOORD0;  // sector's own stream 
};

// maps the local UV to global [0..1] UV
float4 GetUV(float2 UV)
{
  float4 result;
  result.xy = UV*UVScaleOfs.xy+UVScaleOfs.zw;
  result.zw = UV*UVSectorScaleOfs.xy+UVSectorScaleOfs.zw;
  return result;
}

// calculates the world space vertex position
float4 GetVertexPosition(VS_IN In)
{
  float4 pos = float4(In.ObjPosXY.x,In.ObjPosXY.y,In.ObjPosZ,1.0);
  pos.xy = pos.xy*SectorScaleOfs.xy + SectorScaleOfs.zw;
  return pos;
}

#else
struct VS_IN
{ 
  float3 ObjPos : POSITION;   
  float3 Normal : NORMAL;
#if defined(USE_SHADOWMAP)
  float2 UV1 : TEXCOORD1;
#endif
};
#endif

struct VS_OUT
{
  float4 ProjPos : SV_Position;      
  float4 ScreenPos : TEXCOORD0; 
#ifndef USE_SHADOWMAP
  float3 LightDir : TEXCOORD1;
  float3 Normal : TEXCOORD2;
#else
  float2 UV1 : TEXCOORD3;
#endif

#ifdef USE_FOG
  float fogCoord : TEXCOORD4;
#endif
};

VS_OUT main( VS_IN In )
{
  VS_OUT Out;  
  float Offset = 0.5f;
  
#ifdef TERRAIN
  float4 ObjPos4 = GetVertexPosition(In);
#else
  float4 ObjPos4 = float4(In.ObjPos, 1.0f);  
#endif

  Out.ProjPos = mul(matMVP, ObjPos4);
#ifndef USE_SHADOWMAP
  if (LightDirPos.w > 0.0) //point or spot light
  {
    float4 worldPos = mul(matMtoW, ObjPos4);
    Out.LightDir = normalize(LightDirPos.xyz - worldPos.xyz);
  }
  else //directional light
  {
    Out.LightDir = LightDirPos.xyz;
  }
  Out.Normal = normalize(mul((float3x3)matMtoW, In.Normal));
#endif

  Out.ScreenPos = Out.ProjPos;
  Out.ScreenPos.y *= -1.0f;

#if defined(TERRAIN)
  Out.UV1 = GetUV(In.ObjPosXY).zw * MUVtoLM.xy + MUVtoLM.zw;
#elif defined(USE_SHADOWMAP)
  Out.UV1 = In.UV1.xy * modelUV2LM.xy + modelUV2LM.zw;
#endif

#ifdef USE_FOG
  Out.fogCoord = GetFogCoord(depthFog, Out.ProjPos.z);
#endif  
            
  return Out;                              
}