#include <shaders/StaticLightingMobileHelpers.inc>

#ifdef _VISION_DX11
  Texture2D   BaseTexture         : register(t0);
  sampler     BaseTextureSampler  : register(s0);

  #if defined(USE_NORMALMAP)
    Texture2D   NormalMap         : register(t1);
    sampler     NormalMapSampler  : register(s1);
  #endif

  $ifdef LIGHTMAP
    Texture2D   Lightmap            : register(t2);
    sampler     LightmapSampler     : register(s2);
  $endif
  
  #if defined(USE_CUBEMAP)
    TextureCube CubeMap             : register(t3);
    sampler     CubeMapSampler      : register(s3);
  #endif
#else
  sampler2D   BaseTexture         : register(s0);
  
  #if defined(USE_NORMALMAP)
    sampler2D   NormalMap           : register(s1);
  #endif
  
  $ifdef LIGHTMAP
    sampler2D   Lightmap            : register(s2);
  $endif
  
  #if defined(USE_CUBEMAP)
    samplerCUBE CubeMap             : register(s3);
  #endif
#endif

#ifdef _VISION_DX11
  cbuffer g_GlobalConstantBufferFrame : register (b0)
  {
    float4 depthFogCol : packoffset(c20);  // linear depth fog color
  }
  cbuffer g_GlobalConstantBufferUser : register (b2)
  {
    $if VMATERIAL_BUMP || VMATERIAL_BUMPANIMATED
      float4 BumpStrength : packoffset(c0);
    $endif
	
    $if VMATERIAL_FAKESPECBUMP
      float3 SpecColor : packoffset(c0);
    $endif
  }
  cbuffer g_GlobalConstantBufferLightGrid : register (b3)
  {
    $ifdef LIGHTGRIDSIMPLE
      float4 AmbientColor : packoffset(c0);
      float4 DiffuseColor : packoffset(c1);
    $endif
  }  
#else
  float4 depthFogCol : register(c1);  // linear depth fog color
  
  $ifdef LIGHTGRIDSIMPLE
    float4 AmbientColor : register(c26);
    float4 DiffuseColor : register(c27);
  $endif

  $if VMATERIAL_BUMP || VMATERIAL_BUMPANIMATED
    float4 BumpStrength : register(c32);
  $endif
  
  $if VMATERIAL_FAKESPECBUMP
    float3 SpecColor : register(c32);
  $endif
#endif

FIXED4 ps_main(PS_IN In) : SV_Target
{
  half2 BaseMapCoord = (half2) In.UV0;
  
  // Fetch the tangent space normal from the normal map if required
  #if defined(USE_NORMALMAP)
    $if VMATERIAL_BUMPANIMATED
      half3 BumpNormalTS = half3(vTex2D(NormalMap, NormalMapSampler, In.UV1.xy).xyz) * 2.0h - 1.0h;
      BumpNormalTS = lerp(half3(0.0h, 0.0h, 1.0h), BumpNormalTS, half(BumpStrength.x));
      half3 BumpNormalTS2 = half3(vTex2D(NormalMap, NormalMapSampler, In.UV1.zw).xyz) * 2.0h - 1.0h;
      BumpNormalTS += lerp(half3(0.0, 0.0, 1.0), BumpNormalTS, half(BumpStrength.y));
      BumpNormalTS = normalize(BumpNormalTS);
    $else
      half3 BumpNormalTS = half3(vTex2D(NormalMap, NormalMapSampler, BaseMapCoord).xyz) * 2.0h - 1.0h;
    $endif
  #endif

  // If we need world space normals, transform the normal map value if possible, or take the interpolated normal otherwise
#if defined(USE_NORMALS_PS)
  #if defined(USE_TANGENTSPACE_PS) && defined(USE_NORMALMAP)
    half3 BumpNormalWS = BumpNormalTS.x * In.TangentWS - BumpNormalTS.y * In.BiNormalWS + BumpNormalTS.z * In.NormalWS;
  #else
    half3 BumpNormalWS = In.NormalWS;
  #endif
#endif

  // Base lighting: Use the specified lighting method
$if LIGHTGRIDSIMPLE
  $if VMATERIAL_BUMP
    // For simple bump lighting, we transformed the light direction to tangent space in the VS, so we can use that here.
    FIXED Diffuse = max(FIXED(dot(BumpNormalTS, In.LightDir)), FIXED(0.0));
  $else
	// For all other techniques we either use the geometric normal, or we have to transform the normal map to worldspace anyway,
	// so there is no saving from transforming the light direction in the vertex shader first.
    FIXED Diffuse = max(FIXED(dot(BumpNormalWS, In.LightDir)), FIXED(0.0));
  $endif
  
  FIXED3 LightColor = FIXED3(AmbientColor.xyz) + FIXED3(DiffuseColor.xyz) * Diffuse;
$elif LIGHTMAP
  FIXED3 LightColor = vTex2D(Lightmap, LightmapSampler, In.UV1.xy).xyz * 2.0h;
$elif FULLBRIGHT
  FIXED3 LightColor = FIXED3(1, 1, 1);
$else		// Dynamic only
  FIXED3 LightColor = FIXED3(0, 0, 0);
$endif
  
  // Bump lighting uses a fake global light direction which is transformed to tangent space
  // and then combined with the normal map to simulate the normal map's effect. In lightgrid mode,
  // we have a "proper" diffuse light direction and hence don't have to use the fake direction.
$if VMATERIAL_BUMP && !LIGHTGRIDSIMPLE
  LightColor += FIXED3((dot(BumpNormalTS, In.LightDir) - In.LightDir.z) * BumpStrength.x * LightColor);
$endif

  FIXED4 BaseColor = FIXED4(vTex2D(BaseTexture, BaseTextureSampler, BaseMapCoord));
  
  // Add cubemap reflections to the base color instead of the light color so that they are affected by the static lighting,
  // which lets them better blend in with the rest of the scene.
$if VMATERIAL_CUBEMAP || VMATERIAL_BUMPCUBEMAP
  half3 ReflectionVector = reflect(In.EyeDirWS, BumpNormalWS);
  BaseColor.xyz += vTexCUBE(CubeMap, CubeMapSampler, ReflectionVector).xyz * BaseColor.aaa;
$endif

$if VMATERIAL_FAKESPECBUMP
  half3 ReflectionVector = reflect(In.EyeDirWS, BumpNormalWS);
  BaseColor.xyz += vTexCUBE(CubeMap, CubeMapSampler, ReflectionVector).xyz * SpecColor;
$endif
 
  FIXED3 Color = BaseColor.xyz * LightColor;

$if VMATERIAL_GLOW
  Color = lerp(Color, BaseColor.xyz, BaseColor.aaa);
$endif

$ifndef NOFOG
  Color = GetFogColor(Color, depthFogCol.xyz, In.fogCoord);
$endif

  return FIXED4(Color, BaseColor.w);
}